<?php
/* Copyright (C) 2015  Yassine Belkaid   <y.belkaid@nextconcept.ma>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 *      \file       htdocs/transferorders/class/transferorder.class.php
 *      \ingroup    salaries
 *      \brief		Class for transfer orders
 */

// Put here all includes required by your class file
require_once DOL_DOCUMENT_ROOT . '/core/lib/admin.lib.php';
require_once DOL_DOCUMENT_ROOT.'/core/lib/geturl.lib.php';
require_once DOL_DOCUMENT_ROOT .'/core/class/commonobject.class.php';


/**
 *  Class to manage salary payments
 */
class TransferOrder extends CommonObject
{
	var $id;
	var $ref;

	var $fk_user_author;
	var $fk_account;
	var $payment_date;
	var $createdAt;
	var $payment_salaries;
	public $uid;
    public $lastname;
    public $firstname;
    public $admin;
    public $login;
    public $email;
    public $accountLabel;
	public $accountNumber;
	public $element;
	public $table_element;

	/**
	 *	Constructor
	 *
	 *  @param		DoliDB		$db      Database handler
	 */
	function __construct($db) {
		$this->db 			 = $db;
		$this->element 		 = 'transfer_order';
		$this->table_element = 'transfer_order';
		$this->createdAt 	 = date('Y-m-d H:i:s');
		return 1;
	}

	/**
	 * Update database
	 *
	 * @param   User	$user        	User that modify
	 * @param	int		$notrigger	    0=no, 1=yes (no update trigger)
	 * @return  int         			<0 if KO, >0 if OK
	 */
	function update($to_id) {
		global $conf, $langs;

		$error = 0;

		// Clean parameters
		$this->fk_user_author 	= trim($this->fk_user_author);
		$this->fk_account  		= trim($this->fk_account);
		$this->payment_date 	= trim($this->payment_date);

		// Check parameters
		if ($to_id <= 0 || empty($this->fk_user_author) || $this->fk_user_author <= 0 || $this->fk_account <= 0) {
			$this->error = 'ErrorBadParameter';
			return -1;
		}

		$this->db->begin();

		// Update request
		$sql = "UPDATE ".MAIN_DB_PREFIX. $this->table_element ." SET";
		$sql.= " fk_user_author='". (int)$this->db->escape($this->fk_user_author) ."',";
		$sql.= " fk_account=". (int)$this->db->escape($this->fk_account) .",";
		$sql.= " payment_date=". $this->db->idate($this->payment_date) ."";
		$sql.= " WHERE rowid=".$to_id;

		dol_syslog(get_class($this)."::update", LOG_DEBUG);
		$resql = $this->db->query($sql);
		
		if (!$resql) {
			$this->error="Error ".$this->db->lasterror();
			return -1;
		}

		if (!$error) {
			$this->db->commit();
			return $to_id;
		}
		else {
			$this->db->rollback();
			return -1;
		}
	}

	/**
	 *  Load object in memory from database
	 *
	 *  @param	int		$id         id object
	 *  @param  User	$user       User that load
	 *  @return int         		<0 if KO, >0 if OK
	 */
	function fetch($id, $user=null) {
		global $langs;
		$this->error = 0;

		$sql = "SELECT";
		$sql.= " t.rowid,t.fk_user_author,t.fk_account,t.payment_date,t.createdAt,t.payment_salaries,t.entity,t.label,";
		$sql.= " ba.label as accountLabel, ba.number as accountNumber,";
		$sql.= " u.rowid as uid, u.lastname, u.firstname,u.admin,u.login,u.email";
		$sql.= " FROM ".MAIN_DB_PREFIX. $this->table_element ." as t,";
		$sql.= " ".MAIN_DB_PREFIX."bank_account as ba,";
		$sql.= " ".MAIN_DB_PREFIX."user as u";
		$sql.= " WHERE t.rowid = ".$id." AND t.fk_account = ba.rowid AND t.fk_user_author = u.rowid";

		dol_syslog(get_class($this)."::fetch", LOG_DEBUG);
		$resql = $this->db->query($sql);

		if ($resql) {
			if ($this->db->num_rows($resql)) {
				$obj = $this->db->fetch_object($resql);

				$this->id    			= $obj->rowid;
				$this->ref   			= $obj->rowid;
				$this->label   			= $obj->label;
				$this->fk_user_author 	= $obj->fk_user_author;
				$this->payment_date 	= $obj->payment_date;
				$this->createdAt 		= $obj->createdAt;
				$this->fk_account 		= $obj->fk_account;
				$this->entity 			= $obj->entity;
				$this->uid 				= $obj->uid;
			    $this->lastname   = $obj->lastname;
			    $this->firstname  = $obj->firstname;
			    $this->admin 		= $obj->admin;
			    $this->login 		= $obj->login;
			    $this->email 		= $obj->email;
			    $this->accountLabel  = $obj->accountLabel;
				$this->accountNumber = $obj->accountNumber;
				$this->payment_salaries = unserialize($obj->payment_salaries);
			}

			$this->db->free($resql);

			return $this->id;
		}
		else {
			$this->error="Error ".$this->db->lasterror();
			return -1;
		}
	}

	/**
	 *  Delete object in database
	 *
	 *	@param	User	$user       User that delete
	 *	@return	int					<0 if KO, >0 if OK
	 */
	public function delete($user)
	{
		global $conf, $langs;
		$error=0;

		$sql = "DELETE FROM ".MAIN_DB_PREFIX.$this->table_element ." WHERE rowid=".$this->id;

		dol_syslog(get_class($this)."::delete", LOG_DEBUG);
		$resql = $this->db->query($sql);
		if (! $resql) {
			$this->error="Error ".$this->db->lasterror();
			return -1;
		}

		return 1;
	}

    /**
     *  Create in database
     *
     *  @param      User	$user       User that create
     *  @return     int      			<0 if KO, >0 if OK
     */
	function create($user) {
		global $conf, $langs;
		$error = 0;

		/// Clean parameters
		$this->label 			= trim($this->label);
		$this->fk_user_author 	= (int) trim($this->fk_user_author);
		$this->fk_account  		= (int) trim($this->fk_account);
		$this->payment_date 	= trim($this->payment_date);
		$this->createdAt 		= trim($this->createdAt);

		// Check parameters
		if (empty($this->fk_user_author) || $this->fk_user_author <= 0 || $this->fk_account <= 0) {
			$this->error = 'ErrorBadParameter';
			return -1;
		}

		if(!is_array($this->payment_salaries)) {
			$this->error = 'salariesMustBeArray';
			return -2;
		}

		$this->payment_salaries = serialize($this->payment_salaries);

		$this->db->begin();

		// Insert into llx_payment_salary
		$sql = "INSERT INTO ".MAIN_DB_PREFIX. $this->table_element ." (label, fk_user_author, fk_account, payment_date, payment_salaries, createdAt, entity) ";
		$sql.= " VALUES ('". $this->label ."', ". $this->fk_user_author .", ". $this->fk_account .", '".$this->db->idate($this->payment_date)."', '".$this->payment_salaries."', '". $this->createdAt ."', ".$conf->entity .")";

		dol_syslog(get_class($this)."::create", LOG_DEBUG);
		$result = $this->db->query($sql);
		
		if ($result) {

			$this->id = $this->db->last_insert_id(MAIN_DB_PREFIX.$this->table_element);

			if ($this->id > 0) {
				$this->db->commit();
				return $this->id;
			}
			else {
				$this->db->rollback();
				return -2;
			}
		}
		else {
			$this->error = $this->db->error();
			$this->db->rollback();
			return -1;
		}
	}

	/**
	 *	Send name clicable (with possibly the picto)
	 *
	 *	@param	int		$withpicto		0=No picto, 1=Include picto into link, 2=Only picto
	 *	@param	string	$option			link option
	 *	@return	string					Chaine with URL
	 */
	function getNomUrl($withpicto=0,$option='')
	{
		global $langs;

		$result = '';
        $label  = $langs->trans("ShowTransferOrder").': '.$this->ref;
        $link 	= '<a href="'.dol_buildpath('/transferorders/card.php?id='.$this->id,1).'" title="'.dol_escape_htmltag($label, 1).'" class="classfortooltip">';
		$linkend = '</a>';
		$picto 	 = 'payment';

        if ($withpicto) 					$result .= ($link.img_object($label, $picto, 'class="classfortooltip"').$linkend);
		if ($withpicto && $withpicto != 2) 	$result .= ' ';
		if ($withpicto != 2) 				$result .= $link.$this->ref.$linkend;
		
		return $result;
	}

	/**
	 * Fetch all transfer orders
	 *
	 * @return array
	 */
	function fetchAll() {
		$orders = array();
		$sql = "SELECT t.rowid,t.payment_salaries";
		$sql.= " FROM ".MAIN_DB_PREFIX. $this->table_element ." as t";

		dol_syslog(get_class($this)."::fetchAll", LOG_DEBUG);
		$resql = $this->db->query($sql);

		if ($resql) {
			$num = $this->db->num_rows($resql);
			if ($num) {
				$o = 0;

				while ($o < $num) {
					$obj = $this->db->fetch_object($resql);

					$orders[$obj->rowid] = unserialize($obj->payment_salaries);
					$o++;
				}

				$this->db->free($resql);

				return $orders;
			}

			return $orders;
		}
		else {
			$this->error="Error ".$this->db->lasterror();
			return -1;
		}
	}

	/**
	 * Update PaymentSalaries
	 *
	 * @return int     1 updated, 0 error arised
	 */
	public function updatePaymentSalaries()
	{
		$error = 0;

		// Check parameters
		if ($this->id <= 0) {
			$this->error = 'ErrorBadParameter';
			return 0;
		}

		$this->db->begin();

		// Update request
		$sql = "UPDATE ".MAIN_DB_PREFIX. $this->table_element ." SET";
		$sql.= " payment_salaries='". serialize($this->payment_salaries) ."'";
		$sql.= " WHERE rowid=".$this->id;

		dol_syslog(get_class($this)."::updatePaymentSalaries", LOG_DEBUG);
		$resql = $this->db->query($sql);
		
		if (!$resql) {
			$this->error = "Error ".$this->db->lasterror();
			return 0;
		}

		if (!$error) {
			$this->db->commit();
			return 1;
		}
		else {
			$this->db->rollback();
			return 0;
		}
	}

	/**
	 * Get payement type
	 *
	 * @param integer $type_id id of c_paiement
	 */
	public function getPaymentType($type_id)
	{
		$type = "";
	 	$sql = "SELECT libelle as label";
		$sql.= " FROM ".MAIN_DB_PREFIX."c_paiement";
	    $sql.= " WHERE id = ". $this->db->escape((int) $type_id) ." AND active > 0";

	    $resql = $this->db->query($sql);

	    if ($resql) {
			$obj = $this->db->fetch_object($resql);
	    	
	    	$type = $obj->label;
	    }

	    return $type;
	}

	/**
	 * Update payslip when is generated
	 */
	function updatePayslip($id) {
		$sql = 'UPDATE '.MAIN_DB_PREFIX.'payment_salary SET payslip = 1';
		$sql.= ' WHERE rowid = '. $id;
		$result = $this->db->query($sql);

		if ($result) {
			return 1;
		} else {
			dol_print_error($this->db);
			return -1;
		}
	}

	public function getPaymentSalariesByTransferOrderId()
	{
		global $conf;

		$sql = "SELECT u.rowid as uid, u.lastname, u.firstname, u.login, u.email, u.admin, u.salary as current_salary, u.fk_soc as fk_soc,";
		$sql.= " s.rowid, s.fk_user, s.amount, s.salary, s.label, s.datep as datep, s.datev as datev, s.fk_typepayment as type, s.num_payment,";
		$sql.= " pst.code as payment_code";
		$sql.= " FROM ".MAIN_DB_PREFIX."payment_salary as s";
		$sql.= " LEFT JOIN ".MAIN_DB_PREFIX."c_paiement as pst ON s.fk_typepayment = pst.id,";
		$sql.= " ".MAIN_DB_PREFIX."user as u";
		$sql.= " WHERE u.rowid = s.fk_user";
		$sql.= " AND s.entity = ". $conf->entity ." and s.rowid in (". implode(", ", $this->payment_salaries) .")";
	
		$result =  $this->db->query($sql);
		
		return $result;
	}
}


class transferorderscls extends Commonobject{ 
    
    public function __construct($db){ 
        $this->db = $db;
        return 1;
    }

    public function fetch()
    {
        global $conf, $mysoc, $user, $langs;
        $langs->load('transferorders@transferorders');

		dol_include_once('/transferorders/core/modules/modTransferOrders.class.php');
        $modulrecrut = new modTransferOrders($this->db);

        $nummodul = $modulrecrut->numero;
        $editeurmodul = 'nextgestion.com';

        if (!dolibarr_get_const($this->db,'EDITOR_MODULE_transferorders',0))
            dolibarr_set_const($this->db,'EDITOR_MODULE_transferorders','https://www.'.$editeurmodul,'chaine',0,'',0);
        if (!dolibarr_get_const($this->db,'ID_MODULE_transferorders',0))
            dolibarr_set_const($this->db,'ID_MODULE_transferorders',$nummodul,'chaine',0,'',0);
        if (!dolibarr_get_const($this->db,'DATE_MODULE_transferorders',0))
            dolibarr_set_const($this->db,'DATE_MODULE_transferorders',date('Y-m-d'),'chaine',0,'',0);


        $id_modul      = dolibarr_get_const($this->db,'ID_MODULE_transferorders',0);
        $date_modul    = dolibarr_get_const($this->db,'DATE_MODULE_transferorders',0);
        $editor_modul  = dolibarr_get_const($this->db,'EDITOR_MODULE_transferorders',0);

        $link = urlencode(dol_buildpath('/',2));

        if(!empty($date_modul) && $date_modul <= date('Y-m-d') && !empty($editor_modul) && !empty($id_modul) && !empty($link)){

        	$id_modul = urlencode($id_modul);

            $url = $editor_modul."/suivimod/module/registeruse?mod=".$id_modul."&link=".$link;
            $urldata = getURLContent($url);
            $response = json_decode($urldata['content']);

            if($response && $response->actif == 0){
                dolibarr_set_const($this->db,'ID_MODULE_transferorders', 1, 'chaine',0,'',0);
                unActivateModule("modTransferOrders");
            }elseif($response && $response->actif == 1){
				dolibarr_set_const($this->db,'DATE_MODULE_transferorders', date("Y-m-d", time() + 86400), 'chaine',0,'',0);
			}else{
				dolibarr_set_const($this->db,'DATE_MODULE_transferorders', date("Y-m-d"), 'chaine',0,'',0);
			}
        }
        return 1;
    } 
}