<?php
/**
 * Definition class of hosting module
 *
 * Copyright (C) 2011-2019   Jean-François Ferry  <hello@librethic.io>
 * Copyright (C) 2011        Regis Houssin        <regis@dolibarr.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * @package hosting
 *
 */

/**
 *     Description and activation file for module Hosting
 *
 *  \defgroup   hosting     Module Hosting
 *      \brief      Administrative web hosting.
 *      \file       /hosting/core/modules/modHosting.class.php
 *      \ingroup    hosting
 */
include_once DOL_DOCUMENT_ROOT . "/core/modules/DolibarrModules.class.php";

/**
 *
 * Description and activation class for module Hosting
 *
 * @package modHosting
 */
class modHosting extends DolibarrModules
{
    /**
     * Constructor.
     *
     * @param DoliDB $db Database handler
     */
    public function __construct($db)
    {
        global $langs, $conf;

        $this->db = $db;

        // Id for module (must be unique).
        // Use here a free id (See in Home -> System information -> Dolibarr for list of used modules id).
        $this->numero = 110100;
        // Key text used to identify module (for permissions, menus, etc...)
        $this->rights_class = 'hosting';


        // Family can be 'crm','financial','hr','projects','products','ecm','technic','other'
        // It is used to group modules in module setup page
        $this->family = "hosting";
        // Module position in the family on 2 digits ('01', '10', '20', ...)
        $this->module_position = '01';
        // Gives the possibility to the module, to provide his own family info and position of this family (Overwrite $this->family and $this->module_position. Avoid this)
        $this->familyinfo = array('hosting' => array('position' => '10', 'label' => $langs->trans("HostingModulesFamily")));
        
        
        // Module label (no space allowed), used if translation string 'ModuleXXXName' not found (where XXX is value of numeric property 'numero' of module)
        $this->name = preg_replace('/^mod/i', '', get_class($this));
        // Module description, used if translation string 'ModuleXXXDesc' not found (where XXX is value of numeric property 'numero' of module)
        $this->description = "Administrative web hosting.";
        $this->descriptionlong = "";
        
        $this->editor_name = "Librethic";
        $this->editor_url = "https://librethic.io";
        
        // Possible values for version are: 'development', 'experimental', 'dolibarr' or version
        $this->version = '2.0.14+9';

        // Key used in llx_const table to save module status enabled/disabled (where MYMODULE is value of property name of module in uppercase)
        $this->const_name = 'MAIN_MODULE_' . strtoupper($this->name);

        // Name of png file (without png) used for this module.
        // Png file must be in theme/yourtheme/img directory under name object_pictovalue.png.
        $this->picto = 'host@hosting';

        // Defined all module parts (triggers, login, substitutions, menus, css, etc...)
        // for default path (eg: /mymodule/core/xxxxx) (0=disable, 1=enable)
        // for specific path of parts (eg: /mymodule/core/modules/barcode)
        // for specific css file (eg: /mymodule/css/mymodule.css.php)
        //$this->module_parts = array(
        //                            'triggers' => 0,                                     // Set this to 1 if module has its own trigger directory (core/triggers)
        //                            'login' => 0,                                        // Set this to 1 if module has its own login method directory (core/login)
        //                            'substitutions' => 0,                                // Set this to 1 if module has its own substitution function file (core/substitutions)
        //                            'menus' => 0,                                        // Set this to 1 if module has its own menus handler directory (core/menus)
        //                            'theme' => 0,                                        // Set this to 1 if module has its own theme directory (theme)
        //                            'tpl' => 0,                                          // Set this to 1 if module overwrite template dir (core/tpl)
        //                            'barcode' => 0,                                      // Set this to 1 if module has its own barcode directory (core/modules/barcode)
        //                            'models' => 0,                                       // Set this to 1 if module has its own models directory (core/modules/xxx)
        //                            'css' => array('/mymodule/css/mymodule.css.php'),    // Set this to relative path of css file if module has its own css file
        //                            'js' => array('/mymodule/js/mymodule.js'),          // Set this to relative path of js file if module must load a js on all pages
        //                            'hooks' => array('hookcontext1','hookcontext2')      // Set here all hooks context managed by module
        //                            'dir' => array('output' => 'othermodulename'),      // To force the default directories names
        //                            'workflow' => array('WORKFLOW_MODULE1_YOURACTIONTYPE_MODULE2'=>array('enabled'=>'! empty($conf->module1->enabled) && ! empty($conf->module2->enabled)', 'picto'=>'yourpicto@mymodule')) // Set here all workflow context managed by module
        //                        );
        $this->module_parts = array(
            'models' => 1,
            'hooks' => array('commonobject'),
            'tpl' => 0
        );

        // Data directories to create when module is enabled.
        $this->dirs = array('/hosting','/hosting/temp');

        // Config pages. Put here list of php page names stored in admmin directory used to setup module.
        $this->config_page_url = array("admin_hosting.php@hosting");

        // Dependencies
        $this->hidden = false;			// A condition to hide module
        $this->depends = array(); // List of modules id that must be enabled if this module is enabled
        $this->requiredby = array('modDomain', 'modMonitoring'); // List of modules id to disable if this one is disabled
        $this->conflictwith = array();	// List of module class names as string this module is in conflict with
        $this->phpmin = array(5, 4); // Minimum version of PHP required by module
        $this->need_dolibarr_version = array(7.0); // Minimum version of Dolibarr required by module
        $this->langfiles = array("hosting@hosting");
        
        $this->warnings_activation = array();                     // Warning to show when we activate module. array('always'='text') or array('FR'='textfr','ES'='textes'...)
        $this->warnings_activation_ext = array();                 // Warning to show when we activate an external module. array('always'='text') or array('FR'='textfr','ES'='textes'...)
        //$this->automatic_activation = array('FR'=>'Hosting2WasAutomaticallyActivatedBecauseOfYourCountryChoice');
        //$this->always_enabled = true;								// If true, can't be disabled
        

        // Constants
        $this->const = array(); // List of particular constants to add when module is enabled
        //Example: $this->const=array(0=>array('MODULE_MY_NEW_CONST1','chaine','myvalue','This is a constant to add',0),
        //                            1=>array('MODULE_MY_NEW_CONST2','chaine','myvalue','This is another constant to add',0) );
        //$this->triggers = 1;
        // Onglets
        $this->tabs = array();
        $this->tabs[] = array('data' => 'thirdparty:+tabHosting:Hosting:hosting@hosting:$user->rights->hosting->read:/hosting/host_list.php?socid=__ID__');
       

        // Dictionnaries
        if (! isset($conf->hosting) || ! isset($conf->hosting->enabled)) {
            $conf->hosting=new stdClass();
            $conf->hosting->enabled=0;
        }

        $this->dictionaries = array(
            'langs' => 'hosting@hosting',
            'tabname' => array(MAIN_DB_PREFIX . "c_host_category"),
            'tablib' => array("HostingDictCategory"),
            'tabsql' => array('SELECT f.rowid as rowid, f.code, f.pos, f.label, f.active, f.use_default FROM ' . MAIN_DB_PREFIX . 'c_host_category as f'),
            'tabsqlsort' => array("pos ASC"),
            'tabfield' => array("pos,code,label,use_default"),
            'tabfieldvalue' => array("pos,code,label,use_default"),
            'tabfieldinsert' => array("pos,code,label,use_default"),
            'tabrowid' => array("rowid"),
            'tabcond' => array($conf->hosting->enabled),
        );

        // Boxes
        $this->boxes = array(
            0=>array('file'=>'box_hosts.php@hosting2','note'=>'Widget provided by Hosting','enabledbydefaulton'=>'Home')
        ); // List of boxes


        // Cronjobs
        $this->cronjobs = array(); // List of cron jobs entries to add
        // Example: $this->cronjobs=array(0=>array('label'=>'My label', 'jobtype'=>'method', 'class'=>'MyClass', 'method'=>'myMethod', 'parameters'=>'', 'comment'=>'Comment', 'frequency'=>3600, 'unitfrequency'=>3600),
        //                                1=>array('label'=>'My label', 'jobtype'=>'command', 'command'=>'', 'parameters'=>'', 'comment'=>'Comment', 'frequency'=>3600, 'unitfrequency'=>3600)
        // );

        // Permissions
        $this->rights = array(); // Permission array used by this module
        $r = 0;

        $r++;
        $this->rights[$r][0] = 1601;
        $this->rights[$r][1] = 'View hosts';
        $this->rights[$r][3] = 0;
        $this->rights[$r][4] = 'read';

        $r++;
        $this->rights[$r][0] = 1602;
        $this->rights[$r][1] = 'Add/edit hosts';
        $this->rights[$r][3] = 0;
        $this->rights[$r][4] = 'write';

        $r++;
        $this->rights[$r][0] = 1603;
        $this->rights[$r][1] = 'Delete hosts';
        $this->rights[$r][3] = 0;
        $this->rights[$r][4] = 'delete';

        $r++;
        $this->rights[$r][0] = 1604;
        $this->rights[$r][1] = 'Export hosts';
        $this->rights[$r][3] = 0;
        $this->rights[$r][4] = 'export';

        // Main menu entries
        $this->menus = array(); // List of menus to add
        $r = 0;

        // Add here entries to declare new menus
        // Example to declare the Top Menu entry:
        $this->menu[$r] = array(
            'fk_menu' => 0, // Put 0 if this is a top menu
            'type' => 'top', // This is a Top menu entry
            'titre' => 'Hosting',
            'mainmenu' => 'hosting',
            'leftmenu' => 'hosting', // Use 1 if you also want to add left menu entries using this descriptor. Use 0 if left menu entries are defined in a file pre.inc.php (old school).
            'url' => '/hosting/hosting.php',
            'langs' => 'hosting@hosting', // Lang file to use (without .lang) by module. File must be in langs/code_CODE/ directory.
            'position' => 100,
            'perms' => '$user->rights->hosting->read', // Use 'perms'=>'$user->rights->mymodule->level1->level2' if you want your menu with a permission rules
            'target' => '',
            'user' => 2); // 0=Menu for internal users, 1=external users, 2=both
        $r++;

        // Example to declare a Left Menu entry:
        $this->menu[$r] = array(
            'fk_menu' => 'r=0',
            'type' => 'left', // This is a Left menu entry
            'titre' => 'Hosting',
            'mainmenu' => 'hosting',
            'leftmenu' => 'hosting',
            'url' => '/hosting/hosting.php',
            'langs' => 'hosting@hosting', // Lang file to use (without .lang) by module. File must be in langs/code_CODE/ directory.
            'position' => 101,
            'perms' => '$user->rights->hosting->read', // Use 'perms'=>'$user->rights->mymodule->level1->level2' if you want your menu with a permission rules
            'target' => '',
            'user' => 2); // 0=Menu for internal users, 1=external users, 2=both
        $r++;

        // Example to declare a Left Menu entry:
        $this->menu[$r] = array(
            'fk_menu' => 'r=1',
            'type' => 'left', // This is a Left menu entry
            'titre' => 'HostsList',
            'mainmenu' => 'hosting',
            'leftmenu' => 'hosting',
            'url' => '/hosting/host_list.php',
            'langs' => 'hosting@hosting', // Lang file to use (without .lang) by module. File must be in langs/code_CODE/ directory.
            'position' => 102,
            'perms' => '$user->rights->hosting->read', // Use 'perms'=>'$user->rights->mymodule->level1->level2' if you want your menu with a permission rules
            'target' => '',
            'user' => 2); // 0=Menu for internal users, 1=external users, 2=both
        $r++;

        // Example to declare another Left Menu entry:
        $this->menu[$r] = array(
            'fk_menu' => 'r=1',
            'type' => 'left', // This is a Left menu entry
            'titre' => 'HostAdd',
            'mainmenu' => 'hosting',
            'leftmenu' => 'hosting',
            'url' => '/hosting/host_card.php?action=create',
            'langs' => 'hosting@hosting', // Lang file to use (without .lang) by module. File must be in langs/code_CODE/ directory.
            'position' => 103,
            'perms' => '$user->rights->hosting->write', // Use 'perms'=>'$user->rights->mymodule->level1->level2' if you want your menu with a permission rules
            'target' => '',
            'user' => 2); // 0=Menu for internal users, 1=external users, 2=both
        $r++;

        // Exports
        $r = 1;

        // Example:
        $this->export_code[$r] = $this->rights_class . '_' . $r;
        $this->export_label[$r] = 'Hosts'; // Translation key (used only if key ExportDataset_xxx_z not found)
        $this->export_icon[$r] = 'host@hosting';

        $this->export_permission[$r] = array(array("hosting", "export"));
        $this->export_fields_array[$r] = array('s.rowid' => "IdCompany", 's.nom' => 'CompanyName', 's.address' => 'Address', 's.zip' => 'Zip', 's.town' => 'Town', 's.fk_pays' => 'Country', 's.phone' => 'Phone', 's.siren' => 'ProfId1', 's.siret' => 'ProfId2', 's.ape' => 'ProfId3', 's.idprof4' => 'ProfId4', 's.code_compta' => 'CustomerAccountancyCode', 's.code_compta_fournisseur' => 'SupplierAccountancyCode', 'h.rowid' => "HostId", 'h.label' => "HostName", 'h.datec' => "HostDateC", 'actif' => 'Active', 'h.notes' => "Note");
        $this->export_entities_array[$r] = array('s.rowid' => "company", 's.nom' => 'company', 's.address' => 'company', 's.zip' => 'company', 's.town' => 'company', 's.fk_pays' => 'company', 's.phone' => 'company', 's.siren' => 'company', 's.siret' => 'company', 's.ape' => 'company', 's.idprof4' => 'company', 's.code_compta' => 'company', 's.code_compta_fournisseur' => 'company', 'h.rowid' => "Hosting", 'h.label' => "Hosting", 'h.datec' => "hosting", 'actif' => 'hosting', 'h.notes' => "hosting");
        $this->export_alias_array[$r] = array('s.rowid' => "socid", 's.nom' => 'soc_name', 's.address' => 'soc_adress', 's.zip' => 'soc_zip', 's.town' => 'soc_ville', 's.fk_pays' => 'soc_pays', 's.phone' => 'soc_tel', 's.siren' => 'soc_siren', 's.siret' => 'soc_siret', 's.ape' => 'soc_ape', 's.idprof4' => 'soc_idprof4', 's.code_compta' => 'soc_customer_accountancy', 's.code_compta_fournisseur' => 'soc_supplier_accountancy', 'h.rowid' => "host_id", 'h.label' => "host_label", 'h.datec' => "host_datec", 'actif' => 'host_actif', 'h.notes' => "host_notes");
        $this->export_sql_start[$r] = 'SELECT DISTINCT ';
        $this->export_sql_end[$r] = ' FROM (' . MAIN_DB_PREFIX . 'host as h, ' . MAIN_DB_PREFIX . 'societe as s)';
        //$this->export_sql_end[$r] .=' LEFT JOIN '.MAIN_DB_PREFIX.'product as p on (fd.fk_product = p.rowid)';
        $this->export_sql_end[$r] .= ' WHERE h.fk_soc = s.rowid';
        $r++;
    }

    /**
     *    Function called when module is enabled.
     *    The init function add constants, boxes, permissions and menus (defined in constructor) into Dolibarr database.
     *    It also creates data directories.
     *
     *    @param      string          $options Options when enabling module ('', 'newboxdefonly', 'noboxes')
     *                              'noboxes' = Do not insert boxes
     *                              'newboxdefonly' = For boxes, insert def of boxes only and not boxes activation
     *    @return     int             1 if OK, 0 if KO
     */
    public function init($options = '')
    {
        $sql = array();

        $result = $this->load_tables();

        return $this->_init($sql, $options);
    }

    /**
     *    Function called when module is disabled.
     *    Remove from database constants, boxes and permissions from Dolibarr database.
     *    Data directories are not deleted.
     *
     *     @param      string          $options Options when enabling module ('', 'noboxes')
     *    @return     int             1 if OK, 0 if KO
     */
    public function remove($options = '')
    {
        $sql = array();

        return $this->_remove($sql, $options);
    }

     /**
     * Create tables and keys required by module.
     * Files module.sql and module.key.sql with create table and create keys
     * commands must be stored in directory reldir='/module/sql/'
     * This function is called by this->init
     *
     * @param   string  $reldir Relative directory where to scan files
     * @return  int             <=0 if KO, >0 if OK
     */
    public function load_tables()
    {
        return $this->_load_tables('/hosting/sql/');
    }
}
