<?php
/* Copyright (C) 2007-2011 Laurent Destailleur  <eldy@users.sourceforge.net>
 * Copyright (C) 2012-2019 Jean-François Ferry  <hello@librethic.io>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 *      \file       domain/domain.class.php
 *      \ingroup    hosting domain
 *      \brief      CRUD class file (Create/Read/Update/Delete) to manage internet domain
 */

if (!class_exists('Host')) {
    dol_include_once('/hosting/class/host.class.php');
}

/**
 *      \class      Domain
 *      \brief      Manage Internet domain
 *        \remarks
 */
class Domain extends Host
{

    /**
     * @var string Id to identify managed objects
     */
    public $element = 'domain';

    /**
     * @var string Name of table without prefix where object is stored
     */
    public $table_element = 'domain';

    /**
     * @var int  Does domain support multicompany module ? 0=No test on entity, 1=Test with field entity, 2=Test with link by societe
     */
    public $ismultientitymanaged = 0;
    /**
     * @var int  Does domain support extrafields ? 0=No, 1=Yes
     */
    public $isextrafieldmanaged = 1;
    /**
     * @var string String with name of icon for domain. Must be the part after the 'object_' into object_domain.png
     */
    public $picto = 'domain@domain';

    /**
     *  'type' if the field format.
     *  'label' the translation key.

     *  'enabled' is a condition when the field must be managed.
     *  'visible' says if field is visible in list (Examples: 0=Not visible, 1=Visible on list and create/update/view forms, 2=Visible on list only. Using a negative value means field is not shown by default on list but can be selected for viewing)
     *  'notnull' is set to 1 if not null in database. Set to -1 if we must set data to null if empty ('' or 0).
     *  'index' if we want an index in database.
     *  'foreignkey'=>'tablename.field' if the field is a foreign key (it is recommanded to name the field fk_...).
     *  'position' is the sort order of field.
     *  'searchall' is 1 if we want to search in this field when making a search from the quick search button.
     *  'isameasure' must be set to 1 if you want to have a total on list for this field. Field type must be summable like integer or double(24,8).
     *  'help' is a string visible as a tooltip on field
     *  'comment' is not used. You can store here any text of your choice. It is not used by application.
     *  'default' is a default value for creation (can still be replaced by the global setup of default values)
     *  'showoncombobox' if field must be shown into the label of combobox
     */

    /**
     * @var array  Array with all fields and their property. Do not use it as a static var. It may be modified by constructor.
     */
    public $fields=array(
        'rowid' => array('type'=>'integer', 'label'=>'TechnicalID', 'visible'=>-1, 'enabled'=>1, 'position'=>1, 'notnull'=>1, 'index'=>1, 'comment'=>"Id",),
        'ref' => array('type'=>'varchar(128)', 'label'=>'Ref', 'visible'=>1, 'enabled'=>1, 'position'=>10, 'notnull'=>1, 'index'=>1, 'searchall'=>1, 'comment'=>"Reference of object",),
        'entity' => array('type'=>'integer', 'label'=>'Entity', 'visible'=>0, 'enabled'=>1, 'position'=>20, 'notnull'=>-1, 'index'=>1,),
        'label' => array('type'=>'varchar(255)', 'label'=>'DomainName', 'visible'=>1, 'enabled'=>1, 'position'=>30, 'notnull'=>-1, 'searchall'=>1, 'help'=>"Help text",),
        'status' => array('type'=>'integer', 'label'=>'Status', 'visible'=>1, 'enabled'=>1, 'position'=>1000, 'notnull'=>1, 'index'=>1, 'arrayofkeyval'=>array('0'=>'Draft', '1'=>'Active', '9'=>'Expired', '-1'=>'Cancel')),
        'fk_soc' => array('type'=>'integer:Societe:societe/class/societe.class.php', 'label'=>'ThirdParty', 'visible'=>1, 'enabled'=>1, 'position'=>50, 'notnull'=>-1, 'index'=>1, 'help'=>"LinkToThirparty",),
        'fk_supplier' => array('type'=>'integer:Societe:societe/class/societe.class.php', 'label'=>'Supplier', 'visible'=>1, 'enabled'=>1, 'position'=>50, 'notnull'=>-1, 'index'=>1, 'searchall'=>1, 'help'=>"LinkToThirparty",),
        'fk_product' => array('type'=>'integer:Product:product/class/product.class.php', 'label'=>'DomainOfferNew', 'visible'=>1, 'enabled'=>1, 'position'=>50, 'notnull'=>-1, 'index'=>1, 'searchall'=>1, 'help'=>"LinkToProduct",),
        'fk_renew' => array('type'=>'integer:Product:product/class/product.class.php', 'label'=>'DomainOfferReNew', 'visible'=>1, 'enabled'=>1, 'position'=>50, 'notnull'=>-1, 'index'=>1, 'searchall'=>1, 'help'=>"LinkToProductRenew",),
        'fk_user_creat' => array('type'=>'integer', 'label'=>'UserAuthor', 'visible'=>-2, 'enabled'=>1, 'position'=>510, 'notnull'=>1,),
        //'description' => array('type'=>'text', 'label'=>'Descrption', 'visible'=>-1, 'enabled'=>1, 'position'=>60, 'notnull'=>-1,),
        'note_public' => array('type'=>'html', 'label'=>'NotePublic', 'visible'=>0, 'enabled'=>1, 'position'=>61, 'notnull'=>-1,),
        'note_private' => array('type'=>'html', 'label'=>'NotePrivate', 'visible'=>0, 'enabled'=>1, 'position'=>62, 'notnull'=>-1,),
        'date_creation' => array('type'=>'datetime', 'label'=>'DateCreation', 'visible'=>-2, 'enabled'=>1, 'position'=>500, 'notnull'=>1,),
        'date_registrar' => array('type'=>'datetime', 'label'=>'DomainRegistrationDate', 'visible'=>1, 'enabled'=>1, 'position'=>100, 'notnull'=>1,),
        'date_expiration' => array('type'=>'datetime', 'label'=>'DomainExpirationDate', 'visible'=>1, 'enabled'=>1, 'position'=>101, 'notnull'=>1,),
        'tms' => array('type'=>'timestamp', 'label'=>'DateModification', 'visible'=>-2, 'enabled'=>1, 'position'=>501, 'notnull'=>1,),
        'fk_user_modif' => array('type'=>'integer', 'label'=>'UserModif', 'visible'=>-2, 'enabled'=>1, 'position'=>511, 'notnull'=>-1,),
        'import_key' => array('type'=>'varchar(14)', 'label'=>'ImportId', 'visible'=>-2, 'enabled'=>1, 'position'=>1000, 'notnull'=>-1,),

    );
    
    /**
     * Numeric ID
     *
     * @var int ID
     */
    public $id;
    
    public $ref;
    public $entity;

    /**
     * Label
     *
     * @var string
     */
    public $label;
    
    public $fk_status;
    
    /**
     *
     * @var int
     */
    public $fk_soc; // deprecated

    /**
     * Customer ID
     *
     * @var int
     */
    public $socid;

    /**
     * Supplier ID
     *
     * @var int
     */
    public $fk_supplier;

    /**
     * Product ID for creation
     *
     * @var int
     */
    public $fk_product;

    /**
     * ID product for renew
     *
     * @var int
     */

    public $fk_renew;

    /**
     * Public note
     *
     * @var string
     */
    public $note_public;

    /**
     * Private note
     *
     * @var string
     */
    public $note_private;

    /**
     * Creation date
     *
     * @var string
     */
    public $date_creation = '';

    /**
     * Expiration date
     *
     * @var string
     */
    public $date_expiration = '';


    public $tms;
    public $fk_user_modif;
    public $import_key;

    /**
     * Lines
     *
     * @var type
     */
    public $lines;

    /**
     * Results
     *
     * @var DomainResults
     */
    public $results;

    /**
     * To store creation product info
     *
     * @var mixed
     */
    public $product_create;

    /**
     * To store renew product info
     *
     * @var mixed
     */
    public $product_renew;

    /**
     * Store DNS records for domain
     *
     * @var array
     */
    public $dns_records = array();
    
    /**
     * Store Host ID
     *
     * @var int
     */
    public $hostid;
    
    /**
     * Store host ID
     *
     * @var int
     */
    public $host_type_id;

    /**
     * Constructor
     *
     * @param DoliDb $db Database handler
     */
    public function __construct(DoliDB $db)
    {
        $this->db = $db;
        if (empty($conf->global->MAIN_SHOW_TECHNICAL_ID)) {
            $this->fields['rowid']['visible']=0;
        }
        if (empty($conf->multicompany->enabled)) {
            $this->fields['entity']['enabled']=0;
        }
    }

    /**
     * Create object into database
     *
     * @param User $user      User that creates
     * @param bool $notrigger false=launch triggers after, true=disable triggers
     *
     * @return int             <0 if KO, Id of created object if OK
     */
    public function create(User $user, $notrigger = false)
    {
        return $this->createCommon($user, $notrigger);
    }
    
    /**
     * Clone and object into another one
     *
     * @param User $user   User that creates
     * @param int  $fromid Id of object to clone
     *
     * @return mixed                 New object created, <0 if KO
     */
    public function createFromClone(User $user, $fromid)
    {
        global $hookmanager, $langs;
        $error = 0;
        
        dol_syslog(__METHOD__, LOG_DEBUG);
        
        $object = new self($this->db);
        
        $this->db->begin();
        
        // Load source object
        $object->fetchCommon($fromid);
        // Reset some properties
        unset($object->id);
        unset($object->fk_user_creat);
        unset($object->import_key);
        
        // Clear fields
        $object->ref = "copy_of_".$object->ref;
        $object->title = $langs->trans("CopyOf")." ".$object->title;
        // ...
        
        // Create clone
        $object->context['createfromclone'] = 'createfromclone';
        $result = $object->createCommon($user);
        if ($result < 0) {
            $error++;
            $this->error = $object->error;
            $this->errors = $object->errors;
        }
        
        // End
        if (!$error) {
            $this->db->commit();
            return $object;
        } else {
            $this->db->rollback();
            return -1;
        }
    }
    
    /**
     * Load object in memory from the database
     *
     * @param int    $id  Id object
     * @param string $ref Ref
     *
     * @return int         <0 if KO, 0 if not found, >0 if OK
     */
    public function fetch($id, $ref = null, $morewhere = '')
    {
        $result = $this->fetchCommon($id, $ref, $morewhere);
        if ($result > 0) {
            $this->fetchLines();
        }
        return $result;
    }

    /**
     * Load object in memory from the database
     *
     * @param string $domainName Domain name
     *
     * @return int                     <0 if KO, 0 if not found, >0 if OK
     */
    public function fetchDomain($domainName)
    {
        if (empty($domainName)) {
            return -1;
        }

        $sql = 'SELECT '.$this->getFieldList();
        $sql.= ' FROM '.MAIN_DB_PREFIX.$this->table_element;

        $sql.= ' WHERE label = \''.$domainName.'\'';
        
        $sql.=' LIMIT 1';   // This is a fetch, to be sure to get only one record

        $res = $this->db->query($sql);
        if ($res) {
            $obj = $this->db->fetch_object($res);
            if ($obj) {
                $this->setVarsFromFetchObj($obj);
                return $this->id;
            } else {
                return 0;
            }
        } else {
            $this->error = $this->db->lasterror();
            $this->errors[] = $this->error;
            return -1;
        }
    }
    
    /**
     * Load object lines in memory from the database
     *
     * @return int         <0 if KO, 0 if not found, >0 if OK
     */
    /*public function fetchLines()
     {
     $this->lines=array();
     
     // Load lines with object domainLine
     
     return count($this->lines)?1:0;
     }*/
    
    /**
     * Update object into database
     *
     * @param User $user      User that modifies
     * @param bool $notrigger false=launch triggers after, true=disable triggers
     *
     * @return int             <0 if KO, >0 if OK
     */
    public function update(User $user, $notrigger = false)
    {
        return $this->updateCommon($user, $notrigger);
    }
    
    /**
     * Delete object in database
     *
     * @param User $user      User that deletes
     * @param bool $notrigger false=launch triggers after, true=disable triggers
     *
     * @return int             <0 if KO, >0 if OK
     */
    public function delete(User $user, $notrigger = false)
    {
        return $this->deleteCommon($user, $notrigger);
    }
    
    /**
     *  Return a link to the object card (with optionaly the picto)
     *
     * @param int    $withpicto             Include picto in link (0=No picto, 1=Include picto into link, 2=Only picto)
     * @param string $option                On what the link point to ('nolink', ...)
     * @param int    $notooltip             1=Disable tooltip
     * @param string $morecss               Add more css on link
     * @param int    $save_lastsearch_value -1=Auto, 0=No save of lastsearch_values when clicking, 1=Save lastsearch_values whenclicking
     * @param int    $addlabel              0=Default, 1=Add label into string, >1=Add first chars into string
     *
     * @return string                                String with URL
     */
    public function getNomUrl($withpicto = 0, $option = '', $notooltip = 0, $morecss = '', $save_lastsearch_value = -1, $addlabel = 0)
    {
        global $conf, $langs, $hookmanager;
        
        if (! empty($conf->dol_no_mouse_hover)) {
            $notooltip=1;   // Force disable tooltips
        }
        
        $result = '';
        $sep = ' ';
        
        $label = '<u>' . $langs->trans("Domain") . '</u>';
        $label.= '<br>';
        $label.= '<b>' . $langs->trans('Ref') . ':</b> ' . $this->ref;
        $label.= '<br>';
        $label.= '<b>' . $langs->trans('DomainName') . ':</b> ' . $this->label;
        $label.= '<br>';
        $label.= '<b>' . $langs->trans('DomainRegistrationDate') . ':</b> ' . dol_print_date($this->date_registrar, 'dayhour');
        $label.= '<br>';
        $label.= '<b>' . $langs->trans('DomainExpirationDate') . ':</b> ' . dol_print_date($this->date_expiration, 'dayhour');
        $url = dol_buildpath('/domain/domain_card.php', 1).'?id='.$this->id;
        
        if ($option != 'nolink') {
            // Add param to save lastsearch_values or not
            $add_save_lastsearch_values=($save_lastsearch_value == 1 ? 1 : 0);
            if ($save_lastsearch_value == -1 && preg_match('/list\.php/', $_SERVER["PHP_SELF"])) {
                $add_save_lastsearch_values=1;
            }
            if ($add_save_lastsearch_values) {
                $url.='&save_lastsearch_values=1';
            }
        }
        
        $linkclose='';
        if (empty($notooltip)) {
            if (! empty($conf->global->MAIN_OPTIMIZEFORTEXTBROWSER)) {
                $label=$langs->trans("Showdomain");
                $linkclose.=' alt="'.dol_escape_htmltag($label, 1).'"';
            }
            $linkclose.=' title="'.dol_escape_htmltag($label, 1).'"';
            $linkclose.=' class="classfortooltip'.($morecss?' '.$morecss:'').'"';
        } else {
            $linkclose = ($morecss?' class="'.$morecss.'"':'');
        }
        
        $linkstart = '<a href="'.$url.'"';
        $linkstart.=$linkclose.'>';
        $linkend='</a>';
        
        $result .= $linkstart;
        if ($withpicto) {
            $result.=img_object(($notooltip?'':$label), ($this->picto?$this->picto:'generic'), ($notooltip?(($withpicto != 2) ? 'class="paddingright"' : ''):'class="'.(($withpicto != 2) ? 'paddingright ' : '').'classfortooltip"'), 0, 0, $notooltip?0:1);
        }
        if ($withpicto != 2) {
            $result.= $this->ref;
        }
        $result .= $linkend;
        if ($withpicto != 2) {
            $result.=(($addlabel && $this->label) ? $sep . dol_trunc($this->label, ($addlabel > 1 ? $addlabel : 0)) : '');
        }

        global $action;
        $hookmanager->initHooks(array('domaindao'));
        $parameters=array('id'=>$this->id, 'getnomurl'=>$result);
        $reshook = $hookmanager->executeHooks('getNomUrl', $parameters, $this, $action);    // Note that $action and $object may have been modified by some hooks

        if ($reshook > 0) {
            $result = $hookmanager->resPrint;
        } else {
            $result .= $hookmanager->resPrint;
        }

        return $result;
    }
    
    /**
     * Retourne le libelle du status d'un user (actif, inactif)
     *
     * @param int $mode 0=libelle long, 1=libelle court, 2=Picto + Libelle court, 3=Picto, 4=Picto + Libelle long, 5=Libelle court + Picto
     *
     * @return string                    Label of status
     */
    public function getLibStatut($mode = 0)
    {
        return $this->LibStatut($this->status, $mode);
    }
    
    /**
     *  Return the status
     *
     * @param int $status Id status
     * @param int $mode   0=long label, 1=short label, 2=Picto + short label, 3=Picto, 4=Picto + long label, 5=Short label + Picto, 6=Long label + Picto
     *
     * @return string                        Label of status
     */
    public static function LibStatut($status, $mode = 0)
    {
        global $langs;
        
        if ($mode == 0) {
            $prefix='';
            if ($status == 1) {
                return $langs->trans('Enabled');
            }
            if ($status == 0) {
                return $langs->trans('Disabled');
            }
        }
        if ($mode == 1) {
            if ($status == 1) {
                return $langs->trans('Enabled');
            }
            if ($status == 0) {
                return $langs->trans('Disabled');
            }
        }
        if ($mode == 2) {
            if ($status == 1) {
                return img_picto($langs->trans('Enabled'), 'statut4').' '.$langs->trans('Enabled');
            }
            if ($status == 0) {
                return img_picto($langs->trans('Disabled'), 'statut5').' '.$langs->trans('Disabled');
            }
        }
        if ($mode == 3) {
            if ($status == 1) {
                return img_picto($langs->trans('Enabled'), 'statut4');
            }
            if ($status == 0) {
                return img_picto($langs->trans('Disabled'), 'statut5');
            }
        }
        if ($mode == 4) {
            if ($status == 1) {
                return img_picto($langs->trans('Enabled'), 'statut4').' '.$langs->trans('Enabled');
            }
            if ($status == 0) {
                return img_picto($langs->trans('Disabled'), 'statut5').' '.$langs->trans('Disabled');
            }
        }
        if ($mode == 5) {
            if ($status == 1) {
                return $langs->trans('Enabled').' '.img_picto($langs->trans('Enabled'), 'statut4');
            }
            if ($status == 0) {
                return $langs->trans('Disabled').' '.img_picto($langs->trans('Disabled'), 'statut5');
            }
        }
        if ($mode == 6) {
            if ($status == 1) {
                return $langs->trans('Enabled').' '.img_picto($langs->trans('Enabled'), 'statut4');
            }
            if ($status == 0) {
                return $langs->trans('Disabled').' '.img_picto($langs->trans('Disabled'), 'statut5');
            }
        }
    }
    
    /**
     *    Charge les informations d'ordre info dans l'objet commande
     *
     * @param int $id Id of order
     *
     * @return void
     */
    public function info($id)
    {
        $sql = 'SELECT rowid, date_creation as datec, tms as datem,';
        $sql.= ' fk_user_creat, fk_user_modif';
        $sql.= ' FROM '.MAIN_DB_PREFIX.$this->table_element.' as t';
        $sql.= ' WHERE t.rowid = '.$id;
        $result=$this->db->query($sql);
        if ($result) {
            if ($this->db->num_rows($result)) {
                $obj = $this->db->fetch_object($result);
                $this->id = $obj->rowid;
                if ($obj->fk_user_author) {
                    $cuser = new User($this->db);
                    $cuser->fetch($obj->fk_user_author);
                    $this->user_creation   = $cuser;
                }
                
                if ($obj->fk_user_valid) {
                    $vuser = new User($this->db);
                    $vuser->fetch($obj->fk_user_valid);
                    $this->user_validation = $vuser;
                }
                
                if ($obj->fk_user_cloture) {
                    $cluser = new User($this->db);
                    $cluser->fetch($obj->fk_user_cloture);
                    $this->user_cloture   = $cluser;
                }
                
                $this->date_creation     = $this->db->jdate($obj->datec);
                $this->date_modification = $this->db->jdate($obj->datem);
                $this->date_validation   = $this->db->jdate($obj->datev);
            }
            
            $this->db->free($result);
        } else {
            dol_print_error($this->db);
        }
    }
    
    /**
     * Initialise object with example values
     * Id must be 0 if object instance is a specimen
     *
     * @return void
     */
    public function initAsSpecimen()
    {
        $this->initAsSpecimenCommon();
    }
    
    
    /**
     * Action executed by scheduler
     * CAN BE A CRON TASK
     *
     * @return int            0 if OK, <>0 if KO (this function is used also by cron so only 0 is OK)
     */
    public function doScheduledJob()
    {
        global $conf, $langs;
        
        $this->output = '';
        $this->error='';
        
        dol_syslog(__METHOD__, LOG_DEBUG);
        
        // ...
        
        return 0;
    }


    /**
     * Verify domain name
     * Check if A record is present into DNS
     *
     * @use    checkdnsrr()
     * @return mixed
     */
    public function verify_domain_name()
    {
        if (!strlen($this->label)) {
            return '';
        } else {
            return checkdnsrr($this->label, "A");
        }
    }
    
    /**
     * Load object in memory from the database
     *
     * @param string $sortorder  Sort Order
     * @param string $sortfield  Sort field
     * @param int    $limit      offset limit
     * @param int    $offset     offset limit
     * @param array  $filter     filter array
     * @param string $filtermode filter mode (AND or OR)
     *
     * @return int <0 if KO, >0 if OK
     */
    public function fetchAll($sortorder = 'ASC', $sortfield = 't.rowid', $limit = 0, $offset = 0, array $filter = array(), $filtermode = 'AND')
    {
        dol_syslog(__METHOD__, LOG_DEBUG);
        $sql = 'SELECT';
        $sql .= ' t.rowid,';
        $sql .= ' t.date_creation,';
        $sql .= ' hd.fk_host,';
        $sql .= ' t.fk_soc,';
        $sql .= ' t.fk_supplier,';
        $sql .= ' t.fk_product,';
        $sql .= ' t.fk_renew,';
        $sql .= ' t.label,';
        $sql .= ' t.status,';
        $sql .= ' t.fk_user_creat,';
        $sql .= ' t.date_expiration,';
        $sql .= ' t.date_registrar,';
        $sql .= ' t.note_public';
        $sql .= ' FROM ' . MAIN_DB_PREFIX . $this->table_element . ' as t';
        $sql .= " LEFT JOIN " . MAIN_DB_PREFIX . "host_def as hd ON hd.element_id=t.rowid";
        $sql .= ' WHERE hd.element=\'DOMAIN\'';
        // Manage filter
        $sqlwhere = array();
        if (count($filter) > 0) {
            foreach ($filter as $key => $value) {
                if (strpos($key, 'fk_') || strpos($key, 'entity')) {
                    $sqlwhere[] = $key . ' = \'' . $this->db->escape($value) . '\'';
                } else {
                    $sqlwhere[] = $key . ' LIKE \'%' . $this->db->escape($value) . '%\'';
                }
            }
        }
        if (count($sqlwhere) > 0) {
            $sql .= ' AND ' . implode(' ' . $filtermode . ' ', $sqlwhere);
        }

        if (!empty($sortfield)) {
            $sql .= $this->db->order($sortfield, $sortorder);
        }
        if (!empty($limit)) {
            $sql .= ' ' . $this->db->plimit($limit + 1, $offset);
        }

        $this->results = array();
        $resql = $this->db->query($sql);
        if ($resql) {
            $num = $this->db->num_rows($resql);
            while ($obj = $this->db->fetch_object($resql)) {
                $line = new Domain($this->db);
                $line->id = $obj->rowid;
                $line->date_creation = $this->db->jdate($obj->date_creation);
                $line->fk_soc = $obj->fk_soc;
                $line->fk_host = $obj->fk_host;
                $line->socid = $obj->fk_soc;
                $line->fk_supplier = $obj->fk_supplier;
                $line->fk_product = $obj->fk_product;
                $line->fk_renew = $obj->fk_renew;
                $line->label = $obj->label;
                $line->status = $obj->status;
                $line->fk_user_creat = $obj->fk_user_creat;
                $line->date_registrar = $this->db->jdate($obj->date_registrar);
                $line->date_expiration = $this->db->jdate($obj->date_expiration);
                $line->note = $obj->note;

                $this->lines[] = $line;
            }
            $this->db->free($resql);
            return $num;
        } else {
            $this->errors[] = 'Error ' . $this->db->lasterror();
            dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);
            return -1;
        }
    }
    

    public function fetch_lines()
    {
        return $this->fetchLines();
    }

    /**
     *    Load all detailed lines into this->lines
     *
     *    @return int         1 if OK, < 0 if KO
     */
    public function fetchLines()
    {
        $this->lines = array();
        
        // Load class to avoid PHP errors
        if (!class_exists('Product')) {
            require_once DOL_DOCUMENT_ROOT.'/product/class/product.class.php';
        }

        $sql = 'SELECT d.rowid, d.fk_product, d.fk_renew, d.label,';
        $sql .= ' d.date_creation as date_start, d.date_expiration as date_end,';
        $sql .= ' p.ref as product_ref, p.fk_product_type as fk_product_type, p.label as product_label, p.description as product_desc,';
        $sql .= ' p.price, p.price_ttc, p.price_base_type, p.tva_tx, p.fk_product_type, p.localtax1_tx, p.localtax2_tx';
        $sql .= ' FROM ' . MAIN_DB_PREFIX . 'domain as d';
        $sql .= ' LEFT JOIN ' . MAIN_DB_PREFIX . 'product as p ON d.fk_renew = p.rowid';
        $sql .= ' WHERE d.rowid = ' . $this->id;

        dol_syslog(get_class($this) . '::fetch_lines sql=' . $sql, LOG_DEBUG);
        $result = $this->db->query($sql);
        if ($result) {
            $num = $this->db->num_rows($result);
            $i = 0;
            while ($i < $num) {
                $objp = $this->db->fetch_object($result);
                $line = new DomainLigne($this->db);

                $line->rowid = $objp->rowid;
                $line->desc = $objp->label; // Description line
                $line->product_type = $objp->product_type; // Type of line
                $line->product_ref = $objp->product_ref; // Ref product
                $line->libelle = $objp->product_label; // TODO deprecated
                $line->product_label = $objp->product_label; // Label product
                $line->product_desc = $objp->product_desc; // Description product
                $line->fk_product_type = $objp->fk_product_type; // Type of product
                $line->qty = 1;
                $line->subprice = $objp->price;
                $line->tva_tx = $objp->tva_tx;
                $line->localtax1_tx = $objp->localtax1_tx;
                $line->localtax2_tx = $objp->localtax2_tx;
                $line->remise_percent = $objp->remise_percent;
                $line->fk_remise_except = $objp->fk_remise_except;
                $line->fk_product = $objp->fk_renew;
                $line->date_start = $this->db->jdate($objp->date_start);
                $line->date_end = $this->db->jdate($objp->date_end);
                $line->date_start = $this->db->jdate($objp->date_start);
                $line->date_end = $this->db->jdate($objp->date_end);
                $line->info_bits = $objp->info_bits;
                $line->total_ht = $objp->price;
                $line->total_tva = $objp->total_tva;
                $line->total_localtax1 = $objp->total_localtax1;
                $line->total_localtax2 = $objp->total_localtax2;
                $line->total_ttc = $objp->price_ttc;
                $line->export_compta = $objp->fk_export_compta;
                $line->code_ventilation = $objp->fk_code_ventilation;
                $line->rang = $objp->rang;
                $line->special_code = $objp->special_code;
                $line->fk_parent_line = $objp->fk_parent_line;

                $this->total_ht = $objp->price;
                $this->total_ttc = $objp->price_ttc;
                $this->total_tva = $objp->price_ttc - $objp->price;
                // Ne plus utiliser
                //$line->price            = $objp->price;
                //$line->remise           = $objp->remise;

                $this->lines[$i] = $line;

                $i++;
            }
            $this->db->free($result);
            return 1;
        } else {
            $this->errors[] = 'Error ' . $this->db->lasterror();
            dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);
            return -3;
        }
    }

    /**
     *        Load the create and renew products of object from id $this->fk_product and $this->fk_renew into this->product_create and this->product_renew
     *
     *        @return int                    <0 if KO, >0 if OK
     */
    public function fetch_domain_products()
    {
        if (empty($this->fk_product) && empty($this->fk_renew)) {
            return 0;
        }

        if (!class_exists('Product')) {
            include_once DOL_DOCUMENT_ROOT . '/product/class/product.class.php';
        }

        if (!empty($this->fk_product)) {
            $product_create = new Product($this->db);
            $result = $product_create->fetch($this->fk_product);
            if ($result) {
                $this->product_create = $product_create;
            } else {
                $error++;
            }
        }

        if (!empty($this->fk_renew)) {
            $product_renew = new Product($this->db);
            $result = $product_renew->fetch($this->fk_renew);
            if ($result) {
                $this->product_renew = $product_renew;
            } else {
                $error++;
            }
        }
        return $error;
    }

    /**
     * Load domain which are expired or going to expire
     */
    public function fetchExpiration($show_expired = 0, $socid = 0, $sortfield = 'd.date_expiration', $sortorder = 'ASC', $limit = 0)
    {

        global $conf;

        dol_syslog(__METHOD__, LOG_DEBUG);

        $sql = "SELECT d.rowid , d.label, d.status, d.fk_supplier, d.fk_soc, d.date_creation, d.date_expiration";
        $sql .= ' FROM ' . MAIN_DB_PREFIX . 'domain AS d';
        $sql .= ' WHERE d.entity = '.getEntity($element);
        // Only domain expire in 30 days
        $conf->global->DOMAIN_EXPIRATION_ALERT_DAYS = 30;
        if ($show_expired) {
            $sql .= " AND d.date_expiration < '". $this->db->idate(dol_now()) ."'";
        } else {
            $sql .= " AND d.date_expiration > '". $this->db->idate(dol_now())."'" ;
            $sql .= " AND d.date_expiration < '". $this->db->idate(dol_now() + ($conf->global->DOMAIN_EXPIRATION_ALERT_DAYS * 24 * 3600))."'" ;
        }

        // Filter on socid
        if ($socid > 0) {
            $sql .= " AND d.fk_doc='$socid'";
        }

        $sql .= " ORDER BY $sortfield $sortorder";

        if ($limit > 0) {
            $sql .= $this->db->plimit($limit + 1, $offset);
        }

        $resql = $this->db->query($sql);
        
        if ($resql) {
            $this->line = array();
            $num = $this->db->num_rows($resql);
            $i = 0;

            if ($num) {
                while ($i < $num) {
                    $obj = $this->db->fetch_object($resql);
                    $this->line[$i] = new stdClass();

                    $this->line[$i]->rowid = $obj->rowid;
                    $this->line[$i]->label = $obj->label;
                    $this->line[$i]->fk_supplier = $obj->fk_supplier;
                    $this->line[$i]->fk_soc = $obj->fk_soc;
                    $this->line[$i]->status = $obj->status;
                    $this->line[$i]->date_creation = $this->db->jdate($obj->date_creation);
                    $this->line[$i]->date_expiration = $this->db->jdate($obj->date_expiration);
                    $this->line[$i]->nbjours = $obj->nbjours;
                    $i++;
                }
            }
            $this->db->free($resql);
            return $num;
        } else {
            $this->errors[] = 'Error ' . $this->db->lasterror();
            dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);
            return -1;
        }
    }

    /**
     * Set expiration date for domain
     *
     * @param int $time Timestamp with new expiration date
     * @return int > 0 if ok, < 0 if KO
     */
    public function setDateExpiration($time)
    {

        if ($time > 0) {
            $sql = "UPDATE ".MAIN_DB_PREFIX.$this->table_element." SET date_expiration='".$this->db->idate($time)."' WHERE rowid=".$this->id ;

            $this->db->begin();
            if (! $error) {
                $res = $this->db->query($sql);
                if ($res===false) {
                    $error++;
                    $this->errors[] = $this->db->lasterror();
                }
            }

            // Commit or rollback
            if ($error) {
                $this->db->rollback();
                return -1;
            } else {
                $this->db->commit();
                return $this->id;
            }
        }
        return 0;
    }

    /**
     * Set registration date for domain
     *
     * @param int $time Timestamp with new expiration date
     * @return int > 0 if ok, < 0 if KO
     */
    public function setDateRegistrar($time)
    {

        if ($time > 0) {
            $sql = "UPDATE ".MAIN_DB_PREFIX.$this->table_element." SET date_registrar='".$this->db->idate($time)."' WHERE rowid=".$this->id ;

            $this->db->begin();
            if (! $error) {
                $res = $this->db->query($sql);
                if ($res===false) {
                    $error++;
                    $this->errors[] = $this->db->lasterror();
                }
            }

            // Commit or rollback
            if ($error) {
                $this->db->rollback();
                return -1;
            } else {
                $this->db->commit();
                return $this->id;
            }
        }
        return 0;
    }

    /**
     * Get a default reference
     *
     * @global type $conf
     * @param Societe $thirdparty Object thirdparty
     * @return string   Reference
     */
    public function getDefaultRef($thirdparty = '')
    {
        global $conf;

        $defaultref = '';
        $modele = empty($conf->global->DOMAIN_ADDON) ? 'mod_domain_simple' : $conf->global->DOMAIN_ADDON;
        // Search template files
        $file = '';
        $classname = '';
        $filefound = 0;
        $dirmodels = array_merge(array('/'), (array) $conf->modules_parts['models']);
        foreach ($dirmodels as $reldir) {
            $file = dol_buildpath($reldir . "core/modules/domain/" . $modele . '.php', 0);
            if (file_exists($file)) {
                $filefound = 1;
                $classname = $modele;
                break;
            }
        }

        if ($filefound) {
            $result = dol_include_once($reldir . "core/modules/domain/" . $modele . '.php');
            $modDomain = new $classname;
            $defaultref = $modDomain->getNextValue($thirdparty, $this);
        }
        if (is_numeric($defaultref) && $defaultref <= 0) {
            $defaultref = '';
        }
        return $defaultref;
    }

    /**
     * Load DNS records into $this->dns_records
     *
     * @param  const $type By default, search any records. Can be DNS_A, DNS_CNAME, DNS_HINFO, DNS_MX, DNS_NS, DNS_PTR, DNS_SOA, DNS_TXT, DNS_AAAA, DNS_SRV, DNS_NAPTR, DNS_A6, DNS_ALL or DNS_ANY.
     * @return bool
     */
    public function loadDnsRecords($type = DNS_ANY)
    {
        $result = dns_get_record($this->label, $type);
        $found = false;
        if ($result) {
            $this->dns_records = $result;
            $found = true;
        } else {
            $this->errors[] = 'Error : DNS records not found for type ' . $type;
        }
        return $found;
    }

    /**
     * Load DNS A records into $this->dns_records
     *
     * @return bool True if record found, false otherwise
     */
    public function getARecords()
    {
        return $this->loadDnsRecords(DNS_A);
    }

    /**
     * Load DNS CNAME records into $this->dns_records
     *
     * @return bool True if record found, false otherwise
     */
    public function getCNAMERecords()
    {
        return $this->loadDnsRecords(DNS_CNAME);
    }

    /**
     * Load DNS HINFO records into $this->dns_records
     *
     * @return bool True if record found, false otherwise
     */
    public function getHINFORecords()
    {
        return $this->loadDnsRecords(DNS_HINFO);
    }

    /**
     * Load DNS MX records into $this->dns_records
     *
     * @return bool True if record found, false otherwise
     */
    public function getMXRecords()
    {
        return $this->loadDnsRecords(DNS_MX);
    }

    /**
     * Load DNS NS records into $this->dns_records
     *
     * @return bool True if record found, false otherwise
     */
    public function getNSRecords()
    {
        return $this->loadDnsRecords(DNS_NS);
    }

    /**
     * Load DNS PTR records into $this->dns_records
     *
     * @return bool True if record found, false otherwise
     */
    public function getPTRRecords()
    {
        return $this->loadDnsRecords(DNS_PTR);
    }

    /**
     * Load DNS SOA records into $this->dns_records
     *
     * @return bool True if record found, false otherwise
     */
    public function getSOARecords()
    {
        return $this->loadDnsRecords(DNS_SOA);
    }

    /**
     * Load DNS TXT records into $this->dns_records
     *
     * @return bool True if record found, false otherwise
     */
    public function getTXTRecords()
    {
        return $this->loadDnsRecords(DNS_TXT);
    }

    /**
     * Load DNS AAAA records into $this->dns_records
     *
     * @return bool True if record found, false otherwise
     */
    public function getAAAARecords()
    {
        return $this->loadDnsRecords(DNS_AAAA);
    }

    /**
     * Load DNS SRV records into $this->dns_records
     *
     * @return bool True if record found, false otherwise
     */
    public function getSRVRecords()
    {
        return $this->loadDnsRecords(DNS_SRV);
    }

    /**
     * Load DNS NAPTR records into $this->dns_records
     *
     * @return bool True if record found, false otherwise
     */
    public function getNAPTRRecords()
    {
        return $this->loadDnsRecords(DNS_NAPTR);
    }

    /**
     * Load DNS A6 records into $this->dns_records
     *
     * @return bool True if record found, false otherwise
     */
    public function getA6Records()
    {
        return $this->loadDnsRecords(DNS_A6);
    }

    /**
     * Load DNS A6 records into $this->dns_records
     *
     * @return bool True if record found, false otherwise
     */
    public function getALLRecords()
    {
        return $this->loadDnsRecords(DNS_ALL);
    }
}

/**
 * Used for list display
 */
class DomainResults
{

    /**
     * @var int ID
     */
    public $id;

    /**
     *
     * @var string
     */
    public $datec = '';

    /**
     *
     * @var int
     */
    public $fk_soc; // deprecated
    /**
     *
     * @var int
     */
    public $socid;

    /**
     *
     * @var int
     */
    public $fk_supplier;

    /**
     *
     * @var int
     */
    public $fk_product;

    /**
     *
     * @var int
     */
    public $fk_renew;

    /**
     *
     * @var string
     */
    public $label;

    /**
     *
     * @var int
     */
    public $fk_host;

    /**
     *
     * @var string
     */
    public $date_expiration = '';

    /**
     *
     * @var string
     */
    public $note;

    /**
     *      Constructor
     *
     *      @param DoliDb $db Database handler
     */
    public function __construct(DoliDB $db)
    {
        $this->db = $db;
        return 1;
    }

    /**
     *
     * @param type $withpicto
     * @return type
     */
    public function getNomUrl($withpicto = 1)
    {
        $domstatic = new Domain($this->db);
        $domstatic->id = $this->id;
        $domstatic->label = $this->label;
        return $domstatic->getNomUrl($withpicto);
    }
}

/**
 *    \class          DomainLigne
 *    \brief          Classe permettant la gestion des lignes de factures
 *                    Gere des lignes de la table llx_facturedet
 */
class DomainLigne
{
    public $db;
    public $error;

    public $oldline;

    //! From llx_facturedet
    public $rowid;
    //! Id facture
    public $fk_facture;
    //! Id parent line
    public $fk_parent_line;
    //! Description ligne
    public $desc;
    public $fk_product; // Id of predefined product
    public $product_type = 0; // Type 0 = product, 1 = Service

    public $qty; // Quantity (example 2)
    public $tva_tx; // Taux tva produit/service (example 19.6)
    public $localtax1_tx; // Local tax 1
    public $localtax2_tx; // Local tax 2
    public $subprice; // P.U. HT (example 100)
    public $remise_percent; // % de la remise ligne (example 20%)
    public $fk_remise_except; // Link to line into llx_remise_except
    public $rang = 0;

    public $info_bits = 0; // Liste d'options cumulables:
    // Bit 0:    0 si TVA normal - 1 si TVA NPR
    // Bit 1:    0 si ligne normal - 1 si bit discount (link to line into llx_remise_except)

    public $special_code; // Liste d'options non cumulabels:
    // 1: frais de port
    // 2: ecotaxe
    // 3: ??

    public $origin;
    public $origin_id;

    //! Total HT  de la ligne toute quantite et incluant la remise ligne
    public $total_ht;
    //! Total TVA  de la ligne toute quantite et incluant la remise ligne
    public $total_tva;
    public $total_localtax1; //Total Local tax 1 de la ligne
    public $total_localtax2; //Total Local tax 2 de la ligne
    //! Total TTC de la ligne toute quantite et incluant la remise ligne
    public $total_ttc;

    public $fk_code_ventilation = 0;
    public $fk_export_compta = 0;

    public $date_start;
    public $date_end;

    // Ne plus utiliser
    //var $price;             // P.U. HT apres remise % de ligne (exemple 80)
    //var $remise;            // Montant calcule de la remise % sur PU HT (exemple 20)

    // From llx_product
    public $ref; // Product ref (deprecated)
    public $product_ref; // Product ref
    public $libelle; // Product label (deprecated)
    public $product_label; // Product label
    public $product_desc; // Description produit

    public $skip_update_total; // Skip update price total for special lines

    /**
     *  Constructor
     *
     *  @param DoliDB $db Database handler
     */
    public function __construct($db)
    {
        $this->db = $db;
    }
}
