<?php
/* Copyright (C) 2014-2020	Charlene BENKE		<charlie@patas-monkey.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 *	\file	   htdocs/customtabs/class/customtabs.class.php
 *	\ingroup	custom-tabs
 *	\brief	  File of class to manage tabs
 */
if ( ! class_exists('CommonObjectPatas')) {
	dol_include_once('/customtabs/class/commonobjectpatas.class.php');
}

/**
 *	Class to manage members type
 */
class Customtabs extends CommonObjectPatas
{
	public $table_element = 'customtabs';

	var $rowid;
	var $id;   // id de l'extrafields

	var $label;
	var $fk_statut;			// complement actif ou pas
	var $tablename;			// nom de la table à créer (0=non 1=oui)
	var $element;			// nom de L'élément lié à la table
	var $mode;				// mode d'affichage de la table (fiche=1, liste=2)
	var $files;				// permet d'ajouter des fichiers (0=non 1=oui)
	var $entity;			// entity
	var $fk_parent;			// clé du menu principal
	var $template;			// template de l'onglet
	var $colnameline;		// numero de ligne des nom de colonne du fichier d'import
	var $importenabled;		// l'importation sur cette liste est autorisé
	var $exportenabled;		// l'exportation sur cette liste est autorisé
	var $deleteenabled;		// La suppression de toute les lignes de la liste est autorisé
	var $csvseparator;		// séparateur du fichier CSV à importer
	var $csvenclosure;		// encadrement pour les chaines de caractères
	var $parentname;		// nom du menu principal
	var $parenttablename; 	// nom de la table du parent (pour gérer les onglets

	/**
	 *	Constructor
	 *
	 *	@param 		DoliDB		$db		Database handler
	 */
	function __construct($db)
	{
		$this->db = $db;
		$this->fk_statut = 1;
		$this->picto = "customtabs@customtabs";
	}


	/**
	 *  Fonction qui permet de creer le customtabs
	 *
	 *  @param	  User		$user		User making creation
	 *  @return	 						>0 if OK, < 0 if KO
	 */
	function create($user)
	{
		global $conf;

		$this->statut=trim($this->statut);

		$sql = "INSERT INTO ".MAIN_DB_PREFIX."customtabs (";
		$sql.= "libelle, fk_statut, tablename, element, mode, files, fk_parent,";
		$sql.= "entity) VALUES (";
		$sql.= "'".$this->db->escape($this->label)."'";
		$sql.= ", 0"; // par défaut l'onglet est inactifs
		$sql.= ", '".$this->db->escape($this->tablename)."'";
		$sql.= ", '".$this->db->escape($this->element)."'";
		$sql.= ", ".$this->mode;
		$sql.= ", ".($this->fk_files==-1?" 0":$this->files);
		$sql.= ", ".($this->fk_parent==-1?" Null":$this->fk_parent);
		$sql.= ", ".getEntity('customtabs');
		$sql.= ")";

		dol_syslog(get_class($this)."::create sql=".$sql);
		$result = $this->db->query($sql);
		if ($result) {
			$this->id = $this->db->last_insert_id(MAIN_DB_PREFIX."customtabs");
			return $this->id;
		} else {
			$this->error=$this->db->error().' sql='.$sql;
			return -1;
		}
	}

	/**
	 *  Met a jour en base donnees du template
	 *
	 *	@param		User	$user	Object user making change
	 *  @return		int				>0 if OK, < 0 if KO
	 */
	function updateTemplate($user)
	{
		$sql = "UPDATE ".MAIN_DB_PREFIX."customtabs ";
		$sql.= " SET template= '".$this->db->escape($this->template)."'";
		$sql.= " WHERE rowid =".$this->rowid;
		$result = $this->db->query($sql);
		if ($result)
			return 1;
		else {
			$this->error=$this->db->error().' sql='.$sql;
			return -1;
		}
	}

	/**
	 *  active la fonction d'export pour cette liste
	 *
	 *	@param		User	$user	Object user making change
	 *  @return		int				>0 if OK, < 0 if KO
	 */
	function setExport($enabled, $user)
	{
		$sql = "UPDATE ".MAIN_DB_PREFIX."customtabs ";
		$sql.= " SET exportenabled= ".$enabled;
		$sql.= " WHERE rowid =".$this->rowid;

		$result = $this->db->query($sql);
		if ($result)
			return 1;
		else {
			$this->error=$this->db->error().' sql='.$sql;
			return -1;
		}
	}

	/**
	 *  active la fonction d'import pour cette liste
	 *
	 *	@param		User	$user	Object user making change
	 *  @return		int				>0 if OK, < 0 if KO
	 */
	function setImport($enabled, $user)
	{
		$sql = "UPDATE ".MAIN_DB_PREFIX."customtabs ";
		$sql.= " SET importenabled= ".$enabled;
		$sql.= " WHERE rowid =".$this->rowid;

		$result = $this->db->query($sql);
		if ($result)
			return 1;
		else {
			$this->error=$this->db->error().' sql='.$sql;
			return -1;
		}
	}

	/**
	 *  active la fonction de suppression pour cette liste
	 *
	 *	@param		User	$user	Object user making change
	 *  @return		int				>0 if OK, < 0 if KO
	 */
	function setDelete($enabled, $user)
	{
		$sql = "UPDATE ".MAIN_DB_PREFIX."customtabs ";
		$sql.= " SET deleteenabled= ".$enabled;
		$sql.= " WHERE rowid =".$this->rowid;

		$result = $this->db->query($sql);
		if ($result)
			return 1;
		else {
			$this->error=$this->db->error().' sql='.$sql;
			return -1;
		}
	}

	function delete_line($idobject)
	{
		$sql = "DELETE FROM ".MAIN_DB_PREFIX."cust_".$this->tablename."_extrafields";
		$sql.= " WHERE fk_object =".$idobject;
		//print $sql;
		$result = $this->db->query($sql);
		if ($result)
			return 1;

		$this->error=$this->db->error().' sql='.$sql;
		return -1;
	}


	/**
	 *  active défini le paramétrage de l'import pour le customtabs
	 *
	 *	@param		User	$user	Object user making change
	 *  @return		int				>0 if OK, < 0 if KO
	 */
	function updateImport($user)
	{
		$sql = "UPDATE ".MAIN_DB_PREFIX."customtabs ";
		$sql.= " SET colnameline= ".$this->colnameline;
		$sql.= " , colnamebased = ".$this->colnamebased;
		$sql.= " , csvseparator= '".$this->csvseparator."'";
		$sql.= " , csvenclosure= '".$this->csvenclosure."'";
		$sql.= " WHERE rowid =".$this->rowid;

		$result = $this->db->query($sql);
		if ($result)
			return 1;
		else {
			$this->error=$this->db->error().' sql='.$sql;
			return -1;
		}
	}


	/**
	 *  Met a jour en base donnees du type
	 *
	 *	@param		User	$user	Object user making change
	 *  @return		int				>0 if OK, < 0 if KO
	 */
	function update($user)
	{
		global $conf;
		$this->label=trim($this->label);

		$sql = "UPDATE ".MAIN_DB_PREFIX."customtabs ";
		$sql.= " SET fk_statut = ".$this->fk_statut;
		$sql.= ", libelle = '".$this->db->escape($this->label)."'";
		$sql.= ", mode = ".$this->mode;
		$sql.= ", element = '".$this->element."'";
		$sql.= ", files = ".$this->files;
		$sql.= ", entity = ".$this->entity;

		$sql.= ", fk_parent = ".($this->fk_parent>0?$this->fk_parent:'null');
		$sql.= " WHERE rowid =".$this->rowid;
		$result = $this->db->query($sql);

		if ($result) {
			// on supprime l'onglet si il est present ou pas
			$sql ="DELETE FROM ".MAIN_DB_PREFIX."const";
			$sql.=" WHERE name ='".$this->db->encrypt('MAIN_MODULE_CUSTOMTABS_TABS_'.$this->rowid, 1)."'";
			$this->db->query($sql);

			// print "==".$this->fk_parent;
			// si il y a un onglet à positionner (actif et que ce n'et pas un sous-menu)
			if ($this->fk_statut == 1) {
				if ($this->fk_parent > 0)
					$contacttab="sub-".$this->element;
				else {
					if ($this->element=='commande')
						$contacttab='order';
					else
						$contacttab=$this->element;
				}
				// on paramètre selon le type d'onglet les choix possibles
				switch($this->mode) {
					case 1 :
						$tabinfo =$contacttab.':+customtabs_'.$this->rowid.':'.$this->label;
						$tabinfo.=':"":@customtabs:/customtabs/tabs/card.php?tabsid='.$this->rowid.'&id=__ID__';
						break;
					case 2 :
						$tabinfo =$contacttab.':+customtabs_'.$this->rowid.':'.$this->label;
						$tabinfo.=':"":@customtabs:/customtabs/tabs/list.php?tabsid='.$this->rowid.'&id=__ID__';
						break;
				}
				//if (substr($this->element, 11) == 'categories_'))
				//	$tabinfo.='&type='.substr($this->element, -1);

				$sql = "INSERT INTO ".MAIN_DB_PREFIX."const ";
				$sql.= " (name, type, value, note, visible, entity)";
				$sql.= " VALUES (";
				$sql.= $this->db->encrypt('MAIN_MODULE_CUSTOMTABS_TABS_'.$this->rowid, 1);
				$sql.= ", 'chaine'";
				$sql.= ", ".$this->db->encrypt($tabinfo, 1);
				$sql.= ", null";
				$sql.= ", '0'";
				$sql.= ", ".$this->entity;
				$sql.= ")";

				dol_syslog(get_class($this)."::update insert_const_tabs sql=".$sql);
				$resql=$this->db->query($sql);
			}
			return 1;
		} else {
			$this->error=$this->db->error().' sql='.$sql;
			return -1;
		}
	}

	/**
	 *	Fonction qui permet de supprimer
	 *
	 *	@param	  int		$rowid		Id of member type to delete
	 *  @return		int					>0 if OK, < 0 if KO
	 */
	function delete($rowid)
	{
		$sql = "DELETE FROM ".MAIN_DB_PREFIX."customtabs WHERE rowid = ".$rowid;
		$ressql=$this->db->query($sql);

		// on supprime aussi l'onglet
		$sql="DELETE FROM ".MAIN_DB_PREFIX."const where name =".$this->db->encrypt('MAIN_MODULE_CUSTOMTABS_TABS_'.$rowid, 1);

		$resql=$this->db->query($sql);
		if ($resql) {
			if ($this->db->affected_rows($resql) )
				return 1;
			else
				return 0;
		} else {
			print "Err : ".$this->db->error();
			return 0;
		}
	}

	/**
	 *  Fonction qui permet d'importer une ligne dans un customTabs
	 *
	 *  @param 		int		$rowid		Id of member type to load
	 *  @param 		string	$tablename	table
	 *  @return		int					<0 if KO, >0 if OK
	 */
	function importLine($fieldssource, $arrayrecord, $id)
	{
		$error=0;

		$this->db->begin();

		$sql = "INSERT INTO ".MAIN_DB_PREFIX.$this->table_element."_extrafields (fk_object";
		foreach ($fieldssource as $key => $value) {
			// seulement si le champs est renseigné
			if ($value['colname'] !='')
				$sql.=",".$value['colname'];
		}
		$sql .= ") VALUES (".$id;
		foreach ($fieldssource as $key => $value) {
			//var_dump($value);
			if ($value['colname'] !='') {
				if ($arrayrecord[$key-1]['val']) {
					switch ($value['type']) {
						case 'int':
							$sql.=" ,".$arrayrecord[$key-1]['val'];
							break;
						case 'price':
							$sql.=" ,".price2num($arrayrecord[$key-1]['val']);
							break;
						case 'date':
							// les dates doivent être au format JJ/MM/AAAA
							$tmpdate=dol_mktime(
											-1, -1, -1,
											substr($arrayrecord[$key-1]['val'], 3, 2),
											substr($arrayrecord[$key-1]['val'], 0, 2),
											substr($arrayrecord[$key-1]['val'], 6, 4)
							);
							$sql.=", '".$this->db->idate($tmpdate)."'";
							break;
						case 'link':
							$param_list=array_keys($attributeParam ['options']);
							// 0 : ObjectName
							// 1 : classPath
							$infoFieldList = explode(":", $param_list[0]);
							dol_include_once($infoFieldList[1]);
							$object = new $infoFieldList[0]($this->db);
							if ($value) {
								$object->fetch(0, $value);
								$sql.=", ".$object->id;
							}
							break;
						default :
							//print "===".$value['type']."<br>";
							$sql.=", '".$this->db->escape($arrayrecord[$key-1]['val'])."'";
					}
				}
				else
					$sql.=", null";
			}
		}
		$sql.=")";

		dol_syslog(get_class($this)."::importLine insert sql=".$sql);
		//print $sql;
		$resql = $this->db->query($sql);
		if (! $resql) {
			$this->error=$this->db->lasterror();
			dol_syslog(get_class($this)."::insert ".$this->error, LOG_ERR);
			$this->db->rollback();
			return -1;
		} else {
			$this->db->commit();
			return 1;
		}
	}


	/**
	 *  Fonction qui permet de les infos de table
	 *
	 *  @param 		int		$rowid		Id of member type to load
	 *  @param 		string	$tablename	table
	 *  @return		int					<0 if KO, >0 if OK
	 */
	function fetch($rowid, $tablename='')
	{
		$sql = "SELECT c.rowid, c.libelle, c.mode, c.files, c.tablename, c.template, c.fk_statut,";
		$sql.= " c.fk_parent, c.element, c.colnameline, c.csvseparator, c.entity,";
		$sql.= " c.exportenabled, c.importenabled, c.deleteenabled, c.csvenclosure, c.colnamebased";
		$sql.= " FROM ".MAIN_DB_PREFIX."customtabs as c";
		if ($rowid >0)
			$sql .= " WHERE c.rowid = ".$rowid;
		else
			$sql .= " WHERE c.tablename = '".$tablename."'";
		dol_syslog(get_class($this)."::fetch sql=".$sql);

		$resql=$this->db->query($sql);
		if ($resql) {
			if ($this->db->num_rows($resql)) {
				$obj = $this->db->fetch_object($resql);

				$this->rowid			= $obj->rowid;
				$this->libelle			= $obj->libelle;
				$this->element			= $obj->element;
				$this->fk_statut		= $obj->fk_statut;
				$this->tablename		= $obj->tablename;
				$this->mode				= $obj->mode;
				$this->files			= $obj->files;
				$this->template			= $obj->template;
				$this->entity			= $obj->entity;
				$this->fk_parent		= $obj->fk_parent;
				$this->colnameline 		= $obj->colnameline;
				$this->csvseparator 	= $obj->csvseparator;
				$this->csvenclosure 	= $obj->csvenclosure;
				$this->exportenabled 	= $obj->exportenabled;
				$this->importenabled 	= $obj->importenabled;
				$this->deleteenabled 	= $obj->deleteenabled;
				$this->colnamebased 	= $obj->colnamebased;

				$this->getcustomtabsname();
				$this->table_element="cust_".$obj->tablename;
			}
			return 1;
		} else {
			$this->error=$this->db->lasterror();
			dol_syslog(get_class($this)."::fetch ".$this->error, LOG_ERR);
			return -1;
		}
	}

	// return list of display field associate to customTabs
	function fetch_field() 
	{
		$sql = "SELECT *";
		$sql .= " FROM ".MAIN_DB_PREFIX."customtabs_field ";
		$sql .= " WHERE fk_customtabs = ".$this->rowid;
		$liste_array = array();
		$resql=$this->db->query($sql);
		if ($resql) {
			$nump = $this->db->num_rows($resql);
			if ($nump) {
				$i = 0;
				while ($i < $nump) {
					$obj = $this->db->fetch_object($resql);
					$customtabsarray = array ();
					$customtabsarray['rowid']		 = $obj->rowid;
					$customtabsarray['label']		 = $langs->trans($obj->label);
					$customtabsarray['posfield']	 = $obj->posfield;
					$customtabsarray['perms']		 = $obj->perms;
					$customtabsarray['elementfield'] = $obj->elementfield;
					$customtabsarray['queryfield']	 = $obj->queryfield;
					$customtabsarray['typeoffield']	 = $obj->typeoffield;
					$liste_array[$obj->rowid] = $customtabsarray;
					$i++;
				}
			}
			//var_dump($liste_array);
			return $liste_array;
		} else
			print $this->db->error();
	}



	/**
	 *  Fonction qui permet de recuperer le nom du complement parent
	 *
	 *  @param 		int		$rowid		Id of member type to load
	 *  @return		int					<0 if KO, >0 if OK
	 */
	function getcustomtabsname()
	{
		$this->parentname	="";
		$sql = "SELECT c.libelle, c.tablename";
		$sql .= " FROM ".MAIN_DB_PREFIX."customtabs as c";
		$sql .= " WHERE c.rowid = ".$this->fk_parent;

		dol_syslog(get_class($this)."::getcustomtabsname sql=".$sql, LOG_DEBUG);

		$resql=$this->db->query($sql);
		if ($resql) {
			if ($this->db->num_rows($resql)) {
				$obj = $this->db->fetch_object($resql);
				$this->parentname		= $obj->libelle;
				$this->parenttablename	= $obj->tablename;
			}
		} else {
			$this->error=$this->db->lasterror();
			dol_syslog(get_class($this)."::getcustomtabsname ".$this->error, LOG_ERR);
		}
	}

	/**
	 *		Renvoie nom clickable (avec eventuellement le picto)
	 *
	 *		@param		int		$withpicto		0=Pas de picto, 1=Inclut le picto dans le lien, 2=Picto seul
	 *		@param		int		$maxlen			length max libelle
	 *		@return		string					String with URL
	 */

	// QUESTION : cette fonction est-elle utilisé? le dossier complément n'existe pas?
	function getNomUrl($withpicto=0, $maxlen=0)
	{
		global $langs;

		$result='';

		$lien = '<a href="'.DOL_URL_ROOT.'/customtabs/complement/fiche.php?rowid='.$this->id.'">';
		$lienfin='</a>';

		$picto='group';
		$label=$langs->trans("ShowTypeCard", $this->libelle);

		if ($withpicto) $result.=($lien.img_object($label, $picto).$lienfin);
		if ($withpicto && $withpicto != 2) $result.=' ';
		$result.=$lien.($maxlen?dol_trunc($this->libelle, $maxlen):$this->libelle).$lienfin;
		return $result;
	}

	function getLibStatut($mode=0, $noentities=0)
	{
		return $this->LibStatut($this->fk_statut, $mode, $noentities);
	}

	/**
	 *	Returns the label of a statut
	 *
	 *	@param	  int		$statut	 id statut
	 *	@param	  int		$mode	   
	 *	@return	 string	  		Label
	 */
	 
	function LibStatut($statut, $mode=0, $noentities=0)
	{
		global $langs;
		
		$libStatus = ($statut == 1?"Enabled":"Disabled");
		
		switch($mode)
		{
			case 0:
				if ($noentities == 0)
					return $langs->trans($libStatus);
				else
					return $langs->transnoentities($libStatus);
			break;
			case 1:
			case 6:
				return $langs->trans($libStatus);
			break;

			case 2:
			case 4:
				return img_picto(
					$langs->trans($libStatus), 
					$this->statuts_img[$statut]
				).' '.$langs->trans($libStatus);
			break;
			case 3:
				return img_picto($langs->trans($libStatus), $this->statuts_img[$statut]);
			break;
			case 5:
				return $langs->trans($libStatus).' '.img_picto(
					$langs->trans($libStatus), $this->statuts_img[$statut]
				);
				break;
		}
	}


	/**
	 *		Renvoie la liste des tables
	 *
	 *		@param		string		$elementname		name of element to filter
	 *		@param		integer		$filtermode			mode of element to select (0= all, 1=card, 2=list)
	 *		@param		integer		$statut				statut of element to select (-1= all, 0=inactive, 1=enabled)
	 *		@return		array							String with URL
	 */
	function liste_array($elementname='', $filtermode=0, $fk_statut = -1)
	{
		global $conf, $langs;

		$liste_array = array();

		$sql = "SELECT rowid, libelle, mode, fk_parent, element, tablename, fk_statut, entity ";
		$sql.= " FROM ".MAIN_DB_PREFIX."customtabs";
		$sql.= " WHERE (entity=".$conf->entity." OR entity=0)";
		if ($elementname)
			$sql.= " AND element ='".$elementname."'";
		if ($filtermode > 0)
			$sql.= " AND mode =".$filtermode;
		if ($fk_statut >= 0)
			$sql.= " AND fk_statut =".$fk_statut;
		

		$resql=$this->db->query($sql);
		if ($resql) {
			$nump = $this->db->num_rows($resql);
			if ($nump) {
				$i = 0;
				while ($i < $nump) {
					$obj = $this->db->fetch_object($resql);
					$customtabsarray = array ();
					$customtabsarray['rowid']		= $obj->rowid;
					$customtabsarray['libelle']		= $langs->trans($obj->libelle);
					$customtabsarray['mode']		= $obj->mode;
					$customtabsarray['tablename']	= $obj->tablename;
					$customtabsarray['element']		= $obj->element;
					$customtabsarray['fk_parent']	= $obj->fk_parent;
					$customtabsarray['fk_statut']	= $obj->fk_statut;
					$customtabsarray['entity']		= $obj->entity;
					
					$liste_array[$obj->rowid] = $customtabsarray;
					$i++;
				}
			}
			//var_dump($liste_array);
			return $liste_array;
		} else
			print $this->db->error();
	}

	function setShowTabs($fk_usergroup, $status)
	{
		// activate
		if ($status==1) {
			$sql = "INSERT INTO ".MAIN_DB_PREFIX."customtabsgroup (fk_customtabs, fk_usergroup)";
			$sql .= " VALUES (". $this->rowid .", ".$fk_usergroup.")";
		} else { // desactivate
			$sql = "DELETE FROM ".MAIN_DB_PREFIX."customtabsgroup";
			$sql .= " WHERE fk_customtabs =".$this->rowid;
			$sql .= " AND fk_usergroup =".$fk_usergroup;
		}

		$resql=$this->db->query($sql);
		if ($resql)
			return 1;
		else {
			print "Err : ".$this->db->error();
			return 0;
		}
	}

	function getShowCustomtabs($fk_user)
	{
		// on vérifie qu'il y a des groupes d'utilisateur,
		$sql = "SELECT * FROM ".MAIN_DB_PREFIX."usergroup_user as ugu";
		$resql=$this->db->query($sql);
		if ($resql) {	// si pas de groupe, c'est ouvert à tous
			if ($this->db->num_rows($resql) == 0)
				return 1;
		}


		// on regarde ensuite si on est habilité
		$sql = "SELECT 1 FROM ".MAIN_DB_PREFIX."customtabs_usergroup_rights as cur, ".MAIN_DB_PREFIX."usergroup_user as ugu";
		$sql .= " WHERE fk_customtabs =".$this->rowid;
		$sql .= " AND ugu.fk_user =".$fk_user;
		$sql .= " AND ugu.fk_usergroup = cur.fk_usergroup";
		$resql=$this->db->query($sql);
		if ($resql) {
			if ($this->db->num_rows($resql))
				return 1;
			else
				return 0;
		} else {
			$this->error=$this->db->lasterror();
			dol_syslog(get_class($this)."::fetch ".$this->error, LOG_ERR);
			return -1;
		}
	}

	/**
	 *		Determine user rights on object ressourtype type
	 *
	 *		@param		User	$user			User to determine the specials rights
	 *		@param		int		$ressource_type	ressource type to test
	 *		@param		int		$table_name		If it's a complement we chack regarding table name
	 *		@return		array					Return an array ('edit'=>val,'create'=>val,'delete'=>val);
	 *		where val is 1 if right is ok or 0 if not
	 */

	function getUserSpecialsRights($user)
	{
		global $langs, $conf;

		$array_return = array();

		if (!empty($user->id)) {
			//If user is admin he get all rights by default
			if ($user->admin) {
				$array_return = array('edit'=>1,'create'=>1,'delete'=>1);
			} else {
				require_once DOL_DOCUMENT_ROOT.'/user/class/usergroup.class.php';
				$usr_group = new UserGroup($this->db);
				$group_array=$usr_group->listGroupsForUser($user->id);
				if (is_array($group_array) && count($group_array)>0) {

					$sql = 'SELECT rights FROM '.MAIN_DB_PREFIX.'customtabs_usergroup_rights WHERE fk_customtabs='.$this->rowid;
					$sql .= ' AND fk_usergroup IN ('.implode(", ", array_keys($group_array)).')';
					if (! empty($conf->multicompany->enabled) && $conf->entity == 1 && $user->admin && ! $user->entity)
						$sql.= " AND entity IS NOT NULL";
					else
						$sql.= " AND entity IN (0,".$conf->entity.")";

					dol_syslog(get_class($this).'::getUserSpecialsRights sql='.$sql);
					$resql=$this->db->query($sql);
					if ($resql) {
						$nump = $this->db->num_rows($resql);
						if ($nump) {
							$array_return=array('edit'=>0, 'create'=>0, 'delete'=>0);
							while ($obj = $this->db->fetch_object($resql)) {
								//User in in group that allow creation of this type of ressource
								if (strpos($obj->rights, 'A')!==false) {
									$array_return['create']=1;
								}
								//User in in group that allow update of this type of ressource
								if (strpos($obj->rights, 'U')!==false) {
									$array_return['edit']=1;
								}
								//User in in group that allow delete of this type of ressource
								if (strpos($obj->rights, 'D')!==false) {
									$array_return['delete']=1;
								}
							}
						}
						$this->db->free($resql);
					}
					else
						print $this->db->error();
				}
			}
		}
		return $array_return;
	}

	/**
	 *	Return list of types of notes
	 *
	 *	@param	string		$selected		Preselected type
	 *	@param  string		$htmlname		Name of field in form
	 * 	@return	void
	 */
	function selectparent($selected='', $htmlname='fk_parent', $rowid=0)
	{
		global $user, $langs;

		dol_syslog(get_class($this)."::select_customtabs".$selected.", ".$htmlname, LOG_DEBUG);

		$sql = "SELECT c.rowid, c.libelle";
		$sql .= " FROM ".MAIN_DB_PREFIX."customtabs as c";
		$sql .= " WHERE c.fk_parent is null";
		$sql .= " AND c.mode = 1"; // uniquement les onglets de type fiche (pas les listes)
		if ($rowid  >0) // pour ne pas se sélectionner soit-même
			$sql .= " AND c.rowid <> ".$rowid;
		dol_syslog(get_class($this)."::selectparent sql=".$sql);

		$resql=$this->db->query($sql);
		if ($resql) {
			$nump = $this->db->num_rows($resql);
			if ($nump) {
				print '<select class="flat" name="'.$htmlname.'">';
				print '<option value="-1"';
				if ($selected == -1) print ' selected="selected"';
				print '>&nbsp;</option>';

				$i = 0;
				while ($i < $nump) {
					$obj = $this->db->fetch_object($resql);
					print '<option value="'.$obj->rowid.'"';
					if ($obj->rowid == $selected) print ' selected="selected"';
					print '>'.$obj->libelle.'</option>';
					$i++;
				}
				print '</select>';
			} else  // si pas de complément, par défaut la zone est alimenté à -1
				print $langs->trans("NoParentDefined").'<input type=hidden name="'.$htmlname.'" value=-1>';
		}
	}

		/**
	 *  Function to get extra fields of a member into $this->array_options List
	 *
	 *  @param	int		$rowid			Id of line
	 *  @param  array	$optionsArray   Array resulting of call of extrafields->fetch_name_optionals_label()
	 *  @return	void
	 */
	function fetch_optionalslist($rowid, $optionsArray='')
	{
		if (! is_array($optionsArray)) {
			// optionsArray not already loaded, so we load it
			require_once DOL_DOCUMENT_ROOT.'/core/class/extrafields.class.php';
			$extrafields = new ExtraFields($this->db);
			$optionsArray = $extrafields->fetch_name_optionals_label();
		}

		// Request to get complementary values
		if (count($optionsArray) > 0) {
			$sql = "SELECT rowid";
			foreach ($optionsArray as $name => $label)
				$sql.= ", ".$name;

			$sql.= " FROM ".MAIN_DB_PREFIX.$this->table_element."_extrafields";
			$sql.= " WHERE fk_object = ".$rowid;
			$sql.= " ORDER BY 2"; // on trie sur le premier champs de la liste
			dol_syslog(get_class($this)."::fetch_optionals sql=".$sql, LOG_DEBUG);
			$resql=$this->db->query($sql);
			if ($resql) {
				if ($this->db->num_rows($resql)) {
					while ($tab = $this->db->fetch_object($resql)) {
						$rowidextrafields = $tab->rowid;
						foreach ($tab as $key => $value) {
							if ($key != 'rowid'
								&& $key != 'tms'
								&& $key != 'fk_member') {
								// we can add this attribute to adherent object
								$this->array_options[$rowidextrafields]["options_$key"]=$value;
							}
						}
					}
				}
				$this->db->free($resql);
			}
			else
				dol_print_error($this->db);
		}
	}

	/**
	 *	Add/Update all extra fields values for the current object.
	 *  All data to describe values to insert are stored into
	 *  $this->array_options=array('keyextrafield'=>'valueextrafieldtoadd')
	 *
	 *  @return	void
	 */
	function insertExtraFields_line($id, $line='')
	{
		global $langs;

		$error=0;

		if (! empty($this->array_options)) {
			// Check parameters
			$langs->load('admin');
			require_once DOL_DOCUMENT_ROOT.'/core/class/extrafields.class.php';
			$extrafields = new ExtraFields($this->db);
			$nofillrequired=0;
			$optionsArray = $extrafields->fetch_name_optionals_label($this->table_element);
			foreach ($this->array_options as $key => $value) {

				$attributeKey = substr($key, 8);   // Remove 'options_' prefix
				$attributeKey = substr($attributeKey, 0, -1); // pour enlever la derniere valeur (1)
				$attributeType  = $extrafields->attribute_type [$attributeKey];
				$attributeSize  = $extrafields->attribute_size [$attributeKey];
				$attributeLabel = $extrafields->attribute_label[$attributeKey];
				$attributeParam = $extrafields->attribute_param[$attributeKey];

				if($extrafields->attribute_required[$key] && $value =="")
				{
					$nofillrequired++;
					$error_field_required[] = $value;
				}

				//print $attributeKey.':'.$attributeType."-".$attributeSize."-".$attributeLabel."-".$attributeParam."-<br>";
				switch ($attributeType) {
					case 'int':
						if (!is_numeric($value) && $value!='') {
							$error++; $this->errors[]=$langs->trans("ExtraFieldHasWrongValue", $attributeLabel);
							return -1;
						}
						elseif ($value=='')
							$this->array_options[$key] = null;

						break;
					case 'price':
					case 'double':

						$this->array_options[$key] = price2num($this->array_options[$key]);
						break;
					case 'date':
						if (is_numeric($this->array_options[$key]))
							$this->array_options[$key]=$this->db->idate($this->array_options[$key]);
						else {
							$tmpdate=dol_mktime(
											-1, -1, -1,
											substr($this->array_options[$key], 3, 2),
											substr($this->array_options[$key], 0, 2),
											substr($this->array_options[$key], 6, 4)
							);
							$this->array_options[$key]=$this->db->idate($tmpdate);
						}
						break;
					case 'datetime':
						if (is_numeric($this->array_options[$key]))
							$this->array_options[$key]=$this->db->idate($this->array_options[$key]);
						else {
							//print "===".$this->array_options[$key];
							$tmpdate=dol_mktime(
											substr($this->array_options[$key], 10, 2), -1, -1,
											substr($this->array_options[$key], 3, 2),
											substr($this->array_options[$key], 0, 2),
											substr($this->array_options[$key], 6, 4)
							);
							$this->array_options[$key]=$this->db->idate($tmpdate);
						}

						break;
					case 'link':
						$param_list=array_keys($attributeParam ['options']);
						// 0 : ObjectName
						// 1 : classPath
						$infoFieldList = explode(":", $param_list[0]);
						dol_include_once($infoFieldList[1]);
						$object = new $infoFieldList[0]($this->db);
						if ($value) {
							$object->fetch(0, $value);
							$this->array_options[$key]=$object->id;
						}
						break;
					case 'varchar' :
					case 'text':
						$this->array_options[$key]=$this->db->escape($this->array_options[$key]);
						break;

				}
			}

			if($nofillrequired) {
				$langs->load('errors');
				setEventMessages($langs->trans('ErrorFieldsRequired').' : '.implode(', ',$error_field_required), null, 'errors');
				return -1;
			}

			$this->db->begin();

			$sql = "INSERT INTO ".MAIN_DB_PREFIX.$this->table_element."_extrafields (fk_object";
			foreach ($optionsArray as $key => $value) {
				//$attributeKey = substr($key,8);   // Remove 'options_' prefix
				// Add field of attribut
				if ($extrafields->attribute_type[$key] != 'separate') // Only for other type of separate
					if (!is_array($this->array_options[$key]))
						$sql.=",".$key;
			}
			$sql .= ") VALUES (".$id;
			foreach ($optionsArray as $key => $value) {
				// Add field of attribut
				if ($extrafields->attribute_type[$key] != 'separate') { // Only for other type of separate)
					if (!is_array($this->array_options['options_'.$key.$line])) {
						if ($this->array_options['options_'.$key.$line] != '')
							$sql.=",'".$this->array_options['options_'.$key.$line]."'";
						else
							$sql.=",null";
					}
				}
			}
			$sql.=")";
			//print $sql.'<br>';
			//exit;
			dol_syslog(get_class($this)."::insertExtraFields insert sql=".$sql);
			$resql = $this->db->query($sql);
			if (! $resql) {
				$this->error=$this->db->lasterror();
				dol_syslog(get_class($this)."::update ".$this->error, LOG_ERR);
				$this->db->rollback();
				return -1;
			} else {
				$this->db->commit();
				return 1;
			}
		}
		else return 0;
	}

	/**
	 *	Add/Update all extra fields values for the current object.
	 *  All data to describe values to insert are stored into
	 *  $this->array_options=array('keyextrafield'=>'valueextrafieldtoadd')
	 *
	 *  @return	void
	 */
	function editExtraFields_line($id, $linerowid)
	{
		global $langs;

		$error=0;

		if (! empty($this->array_options)) {
			// Check parameters
			$langs->load('admin');
			require_once DOL_DOCUMENT_ROOT.'/core/class/extrafields.class.php';
			$extrafields = new ExtraFields($this->db);
			$optionsArray = $extrafields->fetch_name_optionals_label($this->table_element);

			foreach ($this->array_options as $key => $value) {
				$attributeKey = substr($key, 8);   // Remove 'options_' prefix
				$attributeType  = $extrafields->attribute_type [$attributeKey];
				$attributeSize  = $extrafields->attribute_size [$attributeKey];
				$attributeLabel = $extrafields->attribute_label[$attributeKey];
				$attributeParam = $extrafields->attribute_param[$attributeKey];

				switch ($attributeType) {
					case 'int':
						if (!is_numeric($value) && $value!='') {
							$error++; $this->errors[]=$langs->trans("ExtraFieldHasWrongValue", $attributeLabel);
							return -1;
						}
						elseif ($value=='')
							$this->array_options[$key] = null;
						break;

					case 'price':
						$this->array_options[$key] = price2num($this->array_options[$key]);
						break;

					case 'date':
						if (is_numeric($this->array_options[$key]))
							 $this->array_options[$key]=$this->db->idate($this->array_options[$key]);
						else
						{
							$tmpdate=dol_mktime(
											-1, -1, -1,
											substr($this->array_options[$key], 3, 2),
											substr($this->array_options[$key], 0, 2),
											substr($this->array_options[$key], 6, 4)
							);
							$this->array_options[$key]=$this->db->idate($tmpdate);
						}
						break;
					case 'datetime':
						if (is_numeric($this->array_options[$key]))
							$this->array_options[$key]=$this->db->idate($this->array_options[$key]);
						break;
					case 'link':
						$paramList=array_keys($attributeParam ['options']);
						// 0 : ObjectName
						// 1 : classPath
						$infoFieldList = explode(":", $paramList[0]);
						dol_include_once($infoFieldList[1]);
						$object = new $infoFieldList[0]($this->db);
						if ($value) {
							$object->fetch(0, $value);
							$this->array_options[$key]=$object->id;
						}
						break;
					case 'varchar' :
					case 'text':
						$this->array_options[$key]=$this->db->escape($this->array_options[$key]);
						break;
				}
			}
			$this->db->begin();

			// on fait un update de la ligne au lieu d'un annule et remplace
			$sql = "UPDATE ".MAIN_DB_PREFIX.$this->table_element."_extrafields";
			$btopset=true;
			foreach ($this->array_options as $key => $value) {
				$attributeKey = substr($key, 8);   // Remove 'options_' prefix

				if ($extrafields->attribute_type[$attributeKey] != 'separate') {
					// Only for other type of separate)
					if (!is_array($this->array_options[$key])) {
						if ($btopset) {
							$sql.= " SET ";
							$btopset=false;
						} else
							$sql.= " , ";

						if ($this->array_options[$key] != '')
							$sql.= $attributeKey ." = '".$this->array_options[$key]."'";
						else
							$sql.= $attributeKey ." = null";
					}
				}
			}
			$sql .= " WHERE rowid = ".$linerowid;

			$resql = $this->db->query($sql);
			if (! $resql) {
				$this->error=$this->db->lasterror();
				print $sql." : ".$this->error;
				dol_syslog(get_class($this)."::update ".$this->error, LOG_ERR);
				$this->db->rollback();
				return -1;
			} else {
				$this->db->commit();
				return 1;
			}
		}
		else
			return 0;
	}


	/**
	 *	Add/Update all extra fields values for the current object.
	 *  All data to describe values to insert are stored into
	 *  $this->array_options=array('keyextrafield'=>'valueextrafieldtoadd')
	 *
	 *  @return	void
	 */
	function deleteExtraFields_line($linerowid)
	{

		$error=0;

		$this->db->begin();

		$sql = "DELETE FROM ".MAIN_DB_PREFIX.$this->table_element."_extrafields WHERE rowid = ".$linerowid;

		dol_syslog(get_class($this)."::deleteExtraFields_line delete sql=".$sql);
		$resql = $this->db->query($sql);
		if (! $resql) {
			$this->error=$this->db->lasterror();
			dol_syslog(get_class($this)."::update ".$this->error, LOG_ERR);
			$this->db->rollback();
			return -1;
		} else {
			$this->db->commit();
			return 1;
		}
	}

	/**
	 *	Export customTabs definition in XML format
	 *
	 *  @return	void
	 */
	function getexporttable($id)
	{
		require_once DOL_DOCUMENT_ROOT.'/core/class/extrafields.class.php';

		$this->fetch($id);
		$tmp.="<?xml version='1.0' encoding='ISO-8859-1'?><customtabs>\n";
		$tmp.="<label>".$this->libelle."</label>\n";
		$tmp.="<element>".$this->element."</element>\n";
		$tmp.="<tablename>".$this->tablename."</tablename>\n";
		$tmp.="<mode>".$this->mode."</mode>\n";
		$tmp.="<files>".$this->files."</files>\n";
		$tmp.="<template>".htmlentities($this->template)."</template>\n";

		// récupération des champs associés au customtabs
		$elementtype="cust_".$this->tablename;
		$extrafields = new ExtraFields($this->db);
		$extrafields->fetch_name_optionals_label($elementtype);
		$tmp.="<customextrafields>\n";
		foreach ($extrafields->attribute_type as $key => $value) {
			$tmp.="\t".'<customextrafield>'."\n";
			$tmp.="\t \t<key>".$key."</key>\n";
			$tmp.="\t \t<label>".$extrafields->attribute_label[$key]."</label>\n";
			$tmp.="\t \t<type>".$extrafields->attribute_type[$key]."</type>\n";
			$tmp.="\t \t<size>".$extrafields->attribute_size[$key]."</size>\n";
			$tmp.="\t \t<unique>".$extrafields->attribute_unique[$key]."</unique>\n";
			$tmp.="\t \t<param>".serialize($extrafields->attribute_param[$key])."</param>\n";
			$tmp.="\t \t<pos>".$extrafields->attribute_pos[$key]."</pos>\n";
			$tmp.="\t \t<required>".$extrafields->attribute_required[$key]."</required>\n";
			$tmp.="\t".'</customextrafield>'."\n";
		}
		$tmp.="</customextrafields>\n";
		$tmp.="</customtabs>\n";
		return $tmp;
	}

	/**
	 *	Export importTabs definition in XML format
	 *
	 *  @return	void
	 */
	function importlist($xml)
	{
		global $langs;
		$mesg = Array();

		require_once DOL_DOCUMENT_ROOT.'/core/class/extrafields.class.php';

		// on récupère le fichier et on le parse
		libxml_use_internal_errors(true);
		$sxe = simplexml_load_string($xml);
		if ($sxe === false) {
			echo "Erreur lors du chargement du XML\n";
			foreach (libxml_get_errors() as $error)
				echo "\t", $error->message;
		}
		else
			$arraydata = json_decode(json_encode($sxe), TRUE);
		$this->label=		$arraydata['label'];
		$this->element=		$arraydata['element'];
		$this->tablename=	$arraydata['tablename'];
		$this->mode=		$arraydata['mode'];
		$this->files=		$arraydata['files'];
		$this->template=	$arraydata['template'];
		$this->fk_parent=	-1; 					// par défaut en import xml, pas de récusivité


		$elementtype="cust_".$this->tablename;
		$extrafields = new ExtraFields($this->db);

		$result  = $this->create($user, true);
		if ($result > 0 ) {
			// la saisie du nom de la table est obligatoire sinon on ne crée pas la table
			if ($this->tablename ) {
				// définition de la table à créer
				$table=MAIN_DB_PREFIX."cust_".$this->tablename.'_extrafields';
				$fields = array (
					'rowid'=> array('type'=>'int', 'value'=>'11', 'null'=>'not null', 'extra'=>'AUTO_INCREMENT'),
					'tms' => array('type'=>'timestamp', 'attribute'=>'on update CURRENT_TIMESTAMP',
					'default'=>'CURRENT_TIMESTAMP', 'null'=>'not null', 'extra'=> 'ON UPDATE CURRENT_TIMESTAMP'),
					'fk_element' => array('type'=>'int', 'value'=>'11', 'null'=>'null'),  // clé de l'élément
					'fk_customtabs_parent' => array('type'=>'int', 'value'=>'11', 'null'=>'null'),
					'fk_object' => array('type'=>'int', 'value'=>'11', 'null'=>'not null'),
					'import_key' => array('type'=>'varchar', 'value'=>'14', 'default'=>'NULL', 'null'=>'null')
				);
				$result=$this->db->DDLCreateTable($table, $fields, 'rowid', 'InnoDB');
			}

			$tblsheets=$arraydata['customextrafields']['customextrafield'];
			if (count($tblsheets) >0) {
				foreach ($tblsheets as $sheet) {
					// Check values
					if (! ($sheet['type'])) {
						$error++;
						$langs->load("errors");
						$this->errors[]=$langs->trans("ErrorFieldRequired", $langs->trans("Type"));
						$action = 'import';
					}

					if ($sheet['type']=='select' && !$sheet['param']) {
						$error++;
						$langs->load("errors");
						$this->errors[]=$langs->trans("ErrorNoValueForSelectType");
						$action = 'import';
					}
					if ($sheet['type']=='sellist' && !$sheet['param']) {
						$error++;
						$langs->load("errors");
						$this->errors[]=$langs->trans("ErrorNoValueForSelectListType");
						$action = 'import';
					}
					if ($sheet['type']=='checkbox' && !$sheet['param']) {
						$error++;
						$langs->load("errors");
						$this->errors[]=$langs->trans("ErrorNoValueForCheckBoxType");
						$action = 'import';
					}
					if ($sheet['type']=='radio' && !$sheet['param']) {
						$error++;
						$langs->load("errors");
						$this->errors[]=$langs->trans("ErrorNoValueForRadioType");
						$action = 'import';
					}
					if ((($sheet['type']=='radio')
						|| ($sheet['type']=='checkbox')
						|| ($sheet['type']=='radio'))
						&& $sheet['param']) {
						// Construct array for parameter (value of select list)
						$parameters = $sheet['param'];
						$parameters_array = explode("\r\n", $sheet['param']);
						foreach ($parameters_array as $param_ligne) {
							if (!empty($param_ligne)) {
								if (preg_match_all('/,/', $param_ligne, $matches)) {
									if (count($matches[0])>1) {
										$error++;
										$langs->load("errors");
										$this->errors[]=$langs->trans("ErrorBadFormatValueList", $param_ligne);
										$action = 'import';
									}
								} else {
									$error++;
									$langs->load("errors");
									$this->errors[]=$langs->trans("ErrorBadFormatValueList", $param_ligne);
									$action = 'import';
								}
							}
						}
					}
					if (! $error) {
						$pos = $sheet['pos'];

						if (isset($sheet["key"]) && preg_match("/^\w[a-zA-Z0-9-_]*$/", $sheet["key"])) {
							$result=$extrafields->addExtraField(
											$sheet["key"], $sheet["label"], $sheet["type"], $pos,
											$sheet["size"], $elementtype, ($sheet["unique"]?1:0),
											($sheet["required"]?1:0), '', unserialize($sheet['param'])
							);
							if ($result <= 0) {
								$error++;
								$this->errors[]=$extrafields->error;
								//var_dump($extrafields);
							}
						} else {
							$error++;
							$langs->load("errors");
							$this->errors[]=$langs->trans(
											"ErrorFieldCanNotContainSpecialCharacters",
											$langs->transnoentities("AttributeCode")
							);
						}
					} else {
						//var_dump($this->errors);
						exit;
					}
				}
			}
		}
		return $result;
	}

	function setOptionalsFromPost_line($extralabels, &$object= null, $onlykey='', $i=0)
	{
		global $extrafields;
		global $_POST, $langs;
		$nofillrequired='';// For error when required field left blank
		$error_field_required = array();

		if (is_array($extralabels)) {
			// Get extra fields
			foreach ($extralabels as $key => $value) {
				if (! empty($onlykey) && $key != $onlykey)
					continue;

				$key_type = $extrafields->attribute_type[$key];
				if ($extrafields->attribute_required[$key] && ! GETPOST("options_".$key.$i, 2) ) {
					$nofillrequired++;
					$error_field_required[] = $value;
				}

				if (in_array($key_type, array('date', 'datetime'))) {
					// Clean parameters
					$value_key=dol_mktime(
									$_POST["options_".$key.$i."hour"], $_POST["options_".$key.$i."min"], 0,
									$_POST["options_".$key.$i."month"], $_POST["options_".$key.$i."day"],
									$_POST["options_".$key.$i."year"]
					);
				} else if (in_array($key_type, array('checkbox'))) {
					$value_arr=GETPOST("options_".$key.$i);
					if (!empty($value_arr))
						$value_key=implode($value_arr, ',');
					else
						$value_key='';
				} else if (in_array($key_type, array('price', 'double'))) {
					$value_arr=GETPOST("options_".$key.$i);
					$value_key=price2num($value_arr);
				} else
					$value_key=GETPOST("options_".$key.$i);
				$object->array_options["options_".$key.$i]=$value_key;
			}
			if ($nofillrequired) {
				$langs->load('errors');
				setEventMessage($langs->trans('ErrorFieldsRequired').' : '.implode(', ', $error_field_required), 'errors');
				return -1;
			} else
				return 1;
		} else
			return 0;
	}
}

class CustomTabsField extends CommonObject
{
	public $table_element = 'customtabs_fields';

	var $rowid;
	var $fk_customtabs;		// customTabs associé au champ
	var $label;				// etiquette du champs (traduit ou non)
	var $posField;			// ordre d'affichage du champ
	var $perms;				// droit d'affichage du champs
	var $elementField;		// le champ de l'élément
	var $queryField;		// une requete sql permettant de récuperer le champs // protection contre l'injection
	var $typeOfField;		// type de formatage du champ si ce n'est pas du text

	/**
	 *	Constructor
	 *
	 *	@param 		DoliDB		$db		Database handler
	 */
	function __construct($db)
	{
		$this->db = $db;
	}

	/**
	 *  Fonction qui permet de creer le customtabs_field
	 *
	 *  @return	 						>0 if OK, < 0 if KO
	 */
	function create()
	{
		global $conf;

		$this->statut=trim($this->statut);

		$sql = "INSERT INTO ".MAIN_DB_PREFIX."customtabs_field (";
		$sql.= "label, fk_customtabs, posfield, perms, elementfield, queryfield, typeoffield";
		$sql.= ") VALUES (";
		$sql.= "'".$this->db->escape($this->label)."'";
		$sql.= ", ".$this->fk_customtabs;
		$sql.= ", ".($this->posField?$this->posField:0);
		$sql.= "'".$this->db->escape($this->perm)."'";
		$sql.= "'".$this->db->escape($this->elementField)."'";
		$sql.= "'".$this->db->escape($this->queryField)."'";
		$sql.= "'".($this->typeOfField!=''?$this->typeOfField:'Text')."'";

		$sql.= ")";

		dol_syslog(get_class($this)."::create sql=".$sql);
		$result = $this->db->query($sql);
		if ($result) {
			$this->id = $this->db->last_insert_id(MAIN_DB_PREFIX."customtabs_field");
			return $this->id;
		} else {
			$this->error=$this->db->error().' sql='.$sql;
			return -1;
		}
	}

	/**
	 *  Met a jour en base donnees du type
	 *
	 *	@param		User	$user	Object user making change
	 *  @return		int				>0 if OK, < 0 if KO
	 */
	function update()
	{
		$this->label=trim($this->label);

		$sql = "UPDATE ".MAIN_DB_PREFIX."customtabs_field ";
		$sql.= " SET fk_customtabs = ".$this->fk_customtabs;
		$sql.= ", libelle = '".$this->db->escape($this->label)."'";
		$sql.= ", mode = ".$this->mode;
		$sql.= ", element = '".$this->element."'";
		$sql.= ", files = ".$this->files;
		$sql.= ", fk_parent = ".($this->fk_parent>0?$this->fk_parent:'null');
		$sql.= " WHERE rowid =".$this->rowid;
		$result = $this->db->query($sql);

		if ($result) {
			return 1;
		} else {
			$this->error=$this->db->error().' sql='.$sql;
			return -1;
		}
	}

	/**
	 *	Fonction qui permet de supprimer
	 *
	 *	@param	  int		$rowid		Id of member type to delete
	 *  @return		int					>0 if OK, < 0 if KO
	 */
	function delete($rowid)
	{
		$sql = "DELETE FROM ".MAIN_DB_PREFIX."customtabs_field WHERE rowid = ".$rowid;
		$ressql=$this->db->query($sql);

		if ($ressql) {
			if ($this->db->affected_rows($resql) )
				return 1;
			else
				return 0;
		} else {
			print "Err : ".$this->db->error();
			return 0;
		}
	}



}