<?php
/* Copyright (C) 2016 Yassine Belkaid  y.belkaid@nextconcept.ma
 * 
 */

/**
 * \file    marches/purchaseorder/purchaseorder.class.php
 * \ingroup marches
 * \brief   This file is purchaseorder CRUD class file
 *
 */

dol_include_once('/btp/nc_under_modules/marches/db/nxdb.class.php');
require_once DOL_DOCUMENT_ROOT .'/compta/bank/class/account.class.php';

/**
 * Class marches
 *
 * Contract Management class
 */
class PurchaseOrder extends NXdb
{
	/**
	 * @var string Error code (or message)
	 * @deprecated
	 * @see marches::errors
	 */
	public $error;
	/**
	 * @var string[] Error codes (or messages)
	 */
	public $errors = array();
	/**
	 * @var string Id to identify managed objects
	 */
	public $element = 'purchaseorder';
	/**
	 * @var string Name of table without prefix where object is stored
	 */
	public $table_element = 'purchaseorder';

	/**
	 * @var int ID
	 */
	public $id;
	public $fk_created_by;
	public $fk_third_party;
	public $ref;
	public $label;
	public $subject;
	public $num;
	public $creation_date;
	public $date_etabli;
	public $budget;
	public $reception_date;
	public $deadline;
	public $country;
	public $state;
	public $city;
	public $statut;
	public $created_at;
	public $updated_at;

	public $now;
	public $lines = array();
 
	/**
	 * Constructor
	 *
	 * @param DoliDb $db Database handler
	 */
	public function __construct(DoliDBMysqli $db)
	{
		$this->db 		  = $db;
		$this->now 		  = new \DateTime("now");
		$this->now 		  = $this->now->format('Y-m-d H:i:s');
		$this->updated_at = $this->now;
		
		return 1;
	}

	/**
	 * Create object into database
	 *
	 * @return int   id of last inserted id, otherwise -1 if error arised 
	 */
	public function create()
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		// Clean parameters
		$this->fk_third_party 		= $this->fk_third_party ?: null;
		$this->fk_created_by  		= $this->fk_created_by ?: null;
		$this->ref 					= $this->ref ?: null;
		$this->label 				= $this->label ?: null;
		$this->subject 				= $this->subject ?: null;
		$this->num 					= $this->num ?: null;
		$this->creation_date 		= $this->creation_date ?: null;
		$this->date_etabli 			= $this->date_etabli ?: null;
		$this->budget 				= $this->budget ?: null;
		$this->reception_date 		= $this->reception_date ?: null;
		$this->deadline 			= $this->deadline ?: null;
		$this->created_at 	  		= $this->now;

		// Insert request
		$sql = 'INSERT INTO ' . MAIN_DB_PREFIX . $this->table_element . ' (fk_third_party, fk_created_by, ref, label, created_at, updated_at, subject, num, creation_date, date_etabli, budget, reception_date, deadline, country, state, city) VALUES (';
		$sql .= $this->fk_third_party.', '.$this->fk_created_by.', "'. $this->db->escape($this->ref) .'", "'. $this->db->escape($this->label) .'", "'.$this->created_at.'", "'.$this->updated_at .'", "'. $this->db->escape($this->subject) .'", "'. $this->db->escape($this->num) .'", "'.$this->creation_date.'", "'.$this->date_etabli.'", '.$this->budget.', "'.$this->reception_date.'", "'.$this->deadline.'", "'. $this->db->escape($this->country) .'", "'. $this->db->escape($this->state) .'", "'. $this->db->escape($this->city) .'")';

		$this->db->begin();
		$resql = $this->db->query($sql);

		if (!$resql) {
			$this->db->rollback();

			$this->error 	= 'Error Concurrents ' . $this->db->lasterror();
			$this->errors[] = $this->error;
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		} else {
			$this->db->commit();

			return $this->getLasInsrtedId();
		}
	}

	/**
	 * Load object in memory from the database
	 *
	 * @param int    $id  Id object
	 * @param string $num num
	 *
	 * @return int <0 if KO, 0 if not found, >0 if OK
	 */
	public function fetch($id)
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		$sql = 'SELECT * FROM ' . MAIN_DB_PREFIX . $this->table_element. ' WHERE rowid = ' . $id;

		$resql = $this->db->query($sql);
		if ($resql) {
			$numrows = $this->db->num_rows($resql);
			if ($numrows) {
				$obj 			  	  = $this->db->fetch_object($resql);
				$this->id 		  	  = $obj->rowid;
				$this->fk_created_by  = $obj->fk_created_by;
				$this->fk_third_party = $obj->fk_third_party;
				$this->ref 				= $obj->ref;
				$this->label 			= $obj->label;
				$this->subject 			= nl2br($obj->subject);
				$this->num 				= $obj->num;
				$this->budget			= $obj->budget;
				$this->country			= $obj->country;
				$this->state			= $obj->state;
				$this->city				= $obj->city;
				$this->creation_date 	= $this->db->jdate($obj->creation_date);
				$this->date_etabli		= $this->db->jdate($obj->date_etabli);
				$this->reception_date	= $this->db->jdate($obj->reception_date);
				$this->deadline			= $this->db->jdate($obj->deadline);
				$this->statut	  	  = $obj->statut;
				$this->updated_at 	  = $this->db->jdate($obj->updated_at);
				$this->created_at 	  = $this->db->jdate($obj->created_at);
			}

			$this->db->free($resql);

			if ($numrows) {
				return 1;
			} else {
				return 0;
			}
		} else {
			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		}
	}

	/**
	 * Load object in memory from the database
	 *
	 * @param string $sortorder Sort Order
	 * @param string $sortfield Sort field
	 * @param int    $limit     offset limit
	 * @param int    $offset    offset limit
	 * @param array  $filter    filter array
	 * @param string $filtermode filter mode (AND or OR)
	 *
	 * @return int <0 if KO, >0 if OK
	 */
	public function fetchAll($sortorder = '', $sortfield = '', $limit = 0, $offset = 0, $filter = '', $status = '')
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		$sql = 'SELECT * FROM ' . MAIN_DB_PREFIX . $this->table_element;

		if (!empty($filter)) {
			$sql .= ' WHERE '. substr($filter, 4). $status;
		} else {
			if (empty($status))
				$sql .= ' WHERE statut = 0';
			else
				$sql .= ' WHERE '. substr($status, 4);
		}

		if (!empty($sortfield)) {
			$sql .= $this->db->order($sortfield, $sortorder);
		}

		$this->lines = array();
		$resql 		 = $this->db->query($sql);

		if ($resql) {
			$num = $this->db->num_rows($resql);

			while ($obj = $this->db->fetch_object($resql)) {
				$line = new stdClass;

				$line->id 		  	  	= $obj->rowid;
				$line->fk_created_by  	= $obj->fk_created_by;
				$line->fk_third_party 	= $obj->fk_third_party;
				$line->ref 				= $obj->ref;
				$line->label 			= $obj->label;
				$line->subject 			= nl2br($obj->subject);
				$line->num 				= $obj->num;
				$line->budget			= $obj->budget;
				$line->country			= $obj->country;
				$line->state			= $obj->state;
				$line->city				= $obj->city;
				$line->creation_date 	= $this->db->jdate($obj->creation_date);
				$line->date_etabli		= $this->db->jdate($obj->date_etabli);
				$line->reception_date	= $this->db->jdate($obj->reception_date);
				$line->deadline			= $this->db->jdate($obj->deadline);
				$line->statut	  	  = $obj->statut;
				$line->updated_at 	  = $this->db->jdate($obj->updated_at);
				$line->created_at 	  = $this->db->jdate($obj->created_at);
				$this->lines[] 	  	  = $line;
			}
			$this->db->free($resql);

			return $num;
		} else {
			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		}
	}

	/**
	 * Update object into database
	 *
	 * @return int <0 if KO, >0 if OK
	 */
	public function update($id, array $data)
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		if (!$id || $id <= 0)
			return false;

		// Update request
		$sql = 'UPDATE ' . MAIN_DB_PREFIX . $this->table_element . ' SET ';

		if (count($data) && is_array($data)) {
			foreach ($data as $key => $val) {
				$val = is_numeric($val) ? $val : '"'. $this->db->escape($val) .'"';
				$sql .= '`'. $key. '` = '. $val .',';
			}
		} else 
			return;

		$sql  = substr($sql, 0, -1);
		$sql .= ' WHERE rowid = ' . (int) $id;

		$this->db->begin();

		$resql = $this->db->query($sql);

		if (!$resql) {
			$this->db->rollback();
			
			$this->error 	= 'Error ' . $this->db->lasterror();
			$this->errors[] = $this->error;
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		} else {
			$this->db->commit();

			return 1;
		}
	}

	/**
	 * Delete object in database
	 *
	 * @param User $user      User that deletes
	 * @param bool $notrigger false=launch triggers after, true=disable triggers
	 *
	 * @return int <0 if KO, >0 if OK
	 */
	public function delete($id)
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		$this->db->begin();

		$sql 	= 'DELETE FROM ' . MAIN_DB_PREFIX . $this->table_element .' WHERE rowid = ' . $id;
		$resql 	= $this->db->query($sql);

		if (!$resql) {
			$this->db->rollback();

			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		} else {
			$this->db->commit();

			return 1;
		}
	}

	/**
     *	Return clicable name (with picto eventually)
     *
     *  @author Yassine Belkaid <y.belkaid@nextconcept.ma>
     *	@param		int			$withpicto		0=_No picto, 1=Includes the picto in the linkn, 2=Picto only
     *	@return		string						String with URL
     */
    public function getNomUrl($withpicto = 0,  $id, $ref = null)
    {
        global $langs;

        $result	= '';
        $ref 	= (null !== $ref) ? $ref : '';
        $id  	= ($id  ?: '');
        $label  = $ref;

        $link 	 = '<a href="'.dol_buildpath('/btp/nc_under_modules/marches/purchaseorder/index.php?id='. $id,2) .'" title="'.dol_escape_htmltag($label, 1).'" class="classfortooltip">';
        $linkend ='</a>';
        $picto   = $this->defaultPicto;

        if ($withpicto) $result.=($link.img_object($label, $picto, 'class="classfortooltip"').$linkend);
        if ($withpicto && $withpicto != 2) $result.=' ';
        if ($withpicto != 2) $result.=$link.$ref.$linkend;
        return $result;
    }

    public function getMarketHeader()
    {
    	global $langs;

    	$thirdParty = '';

		if (isset($this->fk_third_party) && !empty($this->fk_third_party)) {
			$socstatic = new Societe($this->db);
			$socstatic->fetch($this->fk_third_party);
			$thirdParty = $socstatic->getNomUrl(1);
		}

    	$html = '<table class="border" width="100%" data-row-id="0">';
	    $html .= '<tbody>';

	    $html .= '<tr><td class="fieldrequired" width="25%">'.$langs->trans("PONum").'</td><td>';
	    $html .= $this->num;
	    $html .= '</td></tr>';

	    $html .= '<tr><td>'.$langs->trans("Subject").'</td><td>';
	    $html .= nl2br($this->subject);
	    $html .= '</td></tr>';

		$html .= '<tr><td class="fieldrequired">'.$langs->trans("Label").'</td><td>';
	    $html .= ucfirst($this->label);
	    $html .= '</td></tr>';

	    // Thirdparty
	    $html .= '<tr><td>'.$langs->trans("ThirdPartyMarket").'</td><td>';
		$html .= $thirdParty;
	    $html .= '</td></tr>';

	    $html .= '<tr><td class="">'.$langs->trans("Ref").'</td><td>';
	    $html .= ucfirst($this->ref);
	    $html .= '</td></tr>';

	    $html .= '<tr><td>'.$langs->trans("CreationDate").'</td><td>';
	    $html .=  dol_print_date($this->creation_date, 'day') .'</td></tr>';

	    $html .= '<tr><td class="">'.$langs->trans("Country").'</td><td>';
	    $html .= ucfirst($this->country) .'</td></tr>';

	    $html .= '<tr><td class="">'.$langs->trans("Province").'</td><td>';
	    $html .= ucfirst($this->state) .'</td></tr>';

	    $html .= '<tr><td class="">'.$langs->trans("City").'</td><td>';
	    $html .= ucfirst($this->city) .'</td></tr>';

	    $html .= '</tbody>';
	    $html .= '</table>';

		$html .= '</table><br />';

		return $html;
    }
}
