<?php
/* Copyright (C) Yassine Belkaid y.belkaid@nextconcept.ma
 *
 */

/**
 * \file    marches/appel_offre.class.php
 * \ingroup marches
 * \brief   This file is marches CRUD class file
 *
 */

dol_include_once('/marches/db/nxdb.class.php');

/**
 * Class marches
 *
 * Contract Management class
 */
class AppelOffre extends NXdb
{
	/**
	 * @var string Error code (or message)
	 * @deprecated
	 * @see marches::errors
	 */
	public $error;
	/**
	 * @var string[] Error codes (or messages)
	 */
	public $errors = array();
	/**
	 * @var string Id to identify managed objects
	 */
	public $element = 'appel_offre';
	/**
	 * @var string Name of table without prefix where object is stored
	 */
	public $table_element = 'appel_offre';

	/**
	 * @var int ID
	 */
	public $id;
	public $ref;
	public $statut;
	public $fk_created_by;
	public $fk_classification;
	public $fk_third_party;
	public $fk_bank;
	public $created_at;
	public $updated_at;
	public $start_date; // datetime
	public $end_date; // datetime
	public $label;
	public $subject;
	public $city;
	public $province;
	public $region;
	public $date_visit; // datetime
	public $budget;
	public $provisional_amount;
	public $provisional_date; // datetime
	public $type; // 1 -> marche normale, 2 -> marche cadre 

	public $now;
	public $lines = array();
 
	/**
	 * Constructor
	 *
	 * @param DoliDb $db Database handler
	 */
	public function __construct(DoliDBMysqli $db)
	{
		$this->db 		 = $db;
		$this->now 		 = new \DateTime("now");
		$this->now 		 = $this->now->format('Y-m-d H:i:s');
		$this->updated_at = $this->now;
		return 1;
	}

	/**
	 * Create object into database
	 *
	 * @return int   id of last inserted id, otherwise -1 if error arised 
	 */
	public function create()
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		// Clean parameters
		$this->ref 					= $this->ref ? $this->db->escape($this->ref): null;
		$this->fk_created_by 		= $this->fk_created_by ?: 0;
		$this->fk_classification 	= $this->fk_classification ?: 0;
		$this->fk_bank 				= $this->fk_bank ?: 0;
		$this->fk_third_party		= $this->fk_third_party ?: 0;
		$this->type					= $this->type ?: 1;
		$this->created_at 			= $this->now;
		$this->start_date			= null !== $this->start_date ? $this->start_date : $this->now;
		$this->end_date				= null !== $this->end_date ? $this->end_date : null;
		$this->date_visit			= null !== $this->date_visit ? $this->date_visit : null;
		$this->label				= $this->label ? $this->db->escape($this->label) : null;
		$this->subject				= $this->subject ? $this->db->escape($this->subject) : null;
		$this->city				= $this->city ? $this->db->escape($this->city) : null;
		$this->province				= $this->province ? $this->db->escape($this->province) : null;
		$this->region				= $this->region ? $this->db->escape($this->region) : null;
		$this->budget				= $this->budget ? $this->db->escape($this->budget) : 0;
		$this->provisional_amount	= $this->provisional_amount ? $this->db->escape($this->provisional_amount) : 0;
		$this->provisional_date		= null !== $this->provisional_date ? $this->provisional_date : null;

		// Insert request
		$sql = 'INSERT INTO ' . MAIN_DB_PREFIX . $this->table_element . ' (ref, fk_created_by, fk_classification, fk_third_party, fk_bank, created_at, updated_at, start_date, end_date, date_visit, label, subject, budget, provisional_amount, provisional_date, city, province, region, type) VALUES (';
		$sql .= '"'.$this->ref.'", '.$this->fk_created_by.', '.$this->fk_classification.', '. $this->fk_third_party .', '.$this->fk_bank.', "'.$this->created_at.'", "'.$this->updated_at .'", "'.$this->start_date.'", "'.$this->end_date.'", "'.$this->date_visit.'", "'.$this->label.'", "'.$this->subject.'", '.$this->budget.', '.$this->provisional_amount.', "'.$this->provisional_date.'", "'.$this->city.'", "'.$this->province.'", "'.$this->region.'", '.$this->type.')';

		$this->db->begin();
		$resql = $this->db->query($sql);

		if (!$resql) {
			$this->db->rollback();

			$this->errors[] = 'Error Appel d\'offtr ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		} else {
			$this->db->commit();

			return $this->getLasInsrtedId();
		}
	}

	/**
	 * Load object in memory from the database
	 *
	 * @param int    $id  Id object
	 * @param string $ref Ref
	 *
	 * @return int <0 if KO, 0 if not found, >0 if OK
	 */
	public function fetch($id, $ref = null)
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		$sql = 'SELECT * FROM ' . MAIN_DB_PREFIX . $this->table_element;
		
		if (null !== $ref) {
			$sql .= ' WHERE ref = ' . '\'' . $ref . '\'';
		} else {
			$sql .= ' WHERE rowid = ' . $id;
		}

		$resql = $this->db->query($sql);
		if ($resql) {
			$numrows = $this->db->num_rows($resql);
			if ($numrows) {
				$obj 					 = $this->db->fetch_object($resql);
				$this->id 				 = $obj->rowid;
				$this->ref 			 	 = $obj->ref;
				$this->statut  		 	 = $obj->statut;
				$this->fk_created_by 	 = $obj->fk_created_by;
				$this->fk_classification = $obj->fk_classification;
				$this->fk_third_party    = $obj->fk_third_party;
				$this->fk_bank 			 = $obj->fk_bank;
				$this->start_date 		 = $this->db->jdate($obj->start_date);
				$this->end_date 		 = $this->db->jdate($obj->end_date);
				$this->date_visit 		 = $this->db->jdate($obj->date_visit);
				$this->label 			 = $obj->label;
				$this->subject 			 = $obj->subject;
				$this->city 			 = $obj->city;
				$this->province			 = $obj->province;
				$this->region 			 = $obj->region;
				$this->type 			 = $obj->type;
				$this->budget 			 = (float) $obj->budget;
				$this->provisional_amount= (float) $obj->provisional_amount;
				$this->provisional_date  = $this->db->jdate($obj->provisional_date);
				$this->updated_at 		 = $this->db->jdate($obj->updated_at);
				$this->created_at 		 = $this->db->jdate($obj->created_at);
			}

			$this->db->free($resql);

			if ($numrows) {
				return 1;
			} else {
				return 0;
			}
		} else {
			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		}
	}

	/**
	 * Load object in memory from the database
	 *
	 * @param string $sortorder Sort Order
	 * @param string $sortfield Sort field
	 * @param int    $limit     offset limit
	 * @param int    $offset    offset limit
	 * @param array  $filter    filter array
	 * @param string $filtermode filter mode (AND or OR)
	 *
	 * @return int <0 if KO, >0 if OK
	 */
	public function fetchAll($sortorder = '', $sortfield = '', $limit = 0, $offset = 0, $filter = '', $filtermode = 'AND')
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		$sql = 'SELECT * FROM ' . MAIN_DB_PREFIX . $this->table_element;

		if (!empty($filter)) {
			$sql .= ' WHERE '. substr($filter, 4);
		} else {
			$sql .= ' WHERE statut in (0, 1) ';
		}
		
		if (!empty($sortfield)) {
			$sql .= $this->db->order($sortfield, $sortorder);
		}

		$this->lines = array();
		$resql 		 = $this->db->query($sql);

		if ($resql) {
			$num = $this->db->num_rows($resql);

			while ($obj = $this->db->fetch_object($resql)) {
				$line = new stdClass;

				$line->id 				 = $obj->rowid;
				$line->ref 			 	 = $obj->ref;
				$line->statut 			 = $obj->statut;
				$line->fk_created_by 	 = $obj->fk_created_by;
				$line->fk_classification = $obj->fk_classification;
				$line->fk_third_party    = $obj->fk_third_party;
				$line->fk_bank 			 = $obj->fk_bank;
				$line->start_date 		 = $this->db->jdate($obj->start_date);
				$line->end_date 		 = $this->db->jdate($obj->end_date);
				$line->date_visit 		 = $this->db->jdate($obj->date_visit);
				$line->label 			 = $obj->label;
				$line->subject 			 = $obj->subject;
				$line->city 			 = $obj->city;
				$line->province			 = $obj->subject;
				$line->region 			 = $obj->region;
				$line->region 			 = $obj->region;
				$line->budget 			 = (float) $obj->budget;
				$line->provisional_amount= (float) $obj->provisional_amount;
				$line->provisional_date  = $this->db->jdate($obj->provisional_date);
				$line->updated_at 		 = $this->db->jdate($obj->updated_at);
				$line->created_at 		 = $this->db->jdate($obj->created_at);
				$this->lines[] 			 = $line;
			}
			$this->db->free($resql);

			return $num;
		} else {
			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		}
	}

	/**
	 * Update object into database
	 *
	 * @return int <0 if KO, >0 if OK
	 */
	public function update($id, array $data)
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		if (!$id || $id <= 0)
			return false;

		// Update request
		$sql = 'UPDATE ' . MAIN_DB_PREFIX . $this->table_element . ' SET ';

		if (count($data) && is_array($data))
			foreach ($data as $key => $val) {
				$val = is_numeric($val) ? $val : '"'. $val .'"';
				$sql .= '`'. $key. '` = '. $val .',';
			}

		$sql  = substr($sql, 0, -1);
		$sql .= ' WHERE rowid = ' . $id;

		$this->db->begin();

		$resql = $this->db->query($sql);

		if (!$resql) {
			$this->db->rollback();
			
			$this->error 	= 'Error ' . $this->db->lasterror();
			$this->errors[] = $this->error;
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		} else {
			$this->db->commit();

			return 1;
		}
	}

	/**
	 * Delete object in database
	 *
	 * @param User $user      User that deletes
	 * @param bool $notrigger false=launch triggers after, true=disable triggers
	 *
	 * @return int <0 if KO, >0 if OK
	 */
	public function delete()
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		$this->db->begin();

		$sql 	= 'DELETE FROM ' . MAIN_DB_PREFIX . $this->table_element .' WHERE rowid = ' . $this->id;
		$resql 	= $this->db->query($sql);

		if (!$resql) {
			$this->db->rollback();

			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		} else {
			$this->db->commit();

			return 1;
		}
	}

	/**
     * return contract by salary id with filters if available
     *
     * @author Yassine Belkaid <y.belkaid@nextconcept.ma>
     * @return integer
     */
	public function fetchByUser($user_id, $order = '', $filter = '')
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		$sql = 'SELECT * FROM ' . MAIN_DB_PREFIX . $this->table_element .' WHERE fk_created_by = '. (int) $user_id.' AND statut in(0, 1) '. $order;

		// Manage filter
		if (!empty($filter)) {
			$sql .= $filter;
		}

		$this->lines = array();

		$resql = $this->db->query($sql);
		if ($resql) {
			$num = $this->db->num_rows($resql);

			while ($obj = $this->db->fetch_object($resql)) {
				$line = new stdClass;

				$line->id 				 = $obj->rowid;
				$line->ref 			 	 = $obj->ref;
				$line->statut 			 = $obj->statut;
				$line->fk_created_by 	 = $obj->fk_created_by;
				$line->fk_classification = $obj->fk_classification;
				$line->fk_third_party    = $obj->fk_third_party;
				$line->fk_bank 			 = $obj->fk_bank;
				$line->start_date 		 = $this->db->jdate($obj->start_date);
				$line->end_date 		 = $this->db->jdate($obj->end_date);
				$line->date_visit 		 = $this->db->jdate($obj->date_visit);
				$line->label 			 = $obj->label;
				$line->subject 			 = $obj->subject;
				$line->city 			 = $obj->city;
				$line->province			 = $obj->province;
				$line->region 			 = $obj->region;
				$line->type 			 = $obj->type;
				$line->budget 			 = (float) $obj->budget;
				$line->provisional_amount= (float) $obj->provisional_amount;
				$line->provisional_date  = $this->db->jdate($obj->provisional_date);
				$line->updated_at 		 = $this->db->jdate($obj->updated_at);
				$line->created_at 		 = $this->db->jdate($obj->created_at);
				$this->lines[] 			 = $line;
			}
			$this->db->free($resql);

			return $num;
		} else {
			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		}
	}

	/**
     *	Return clicable name (with picto eventually)
     *
     *  @author Yassine Belkaid <y.belkaid@nextconcept.ma>
     *	@param		int			$withpicto		0=_No picto, 1=Includes the picto in the linkn, 2=Picto only
     *	@return		string						String with URL
     */
    public function getNomUrl($withpicto = 0,  $id = null, $ref = null)
    {
        global $langs;

        $result	= '';
        $setRef = (null !== $ref) ? $ref : '';
        $id  	= ($id  ?: '');
        $label  = $langs->trans("Show").': '. $setRef;


        $link 	 = '<a href="'.dol_buildpath('/marches/index.php?id='. $id,2) .'" title="'.dol_escape_htmltag($label, 1).'" class="classfortooltip picto_marche_module">';
        $linkend ='</a>';
        $picto   = $this->defaultPicto;

        if ($withpicto) $result.=($link.img_object($label, $picto, 'class="classfortooltip"').$linkend);
        if ($withpicto && $withpicto != 2) $result.=' ';
        if ($withpicto != 2) $result.=$link.$setRef.$linkend;
        return $result;
    }
	
	/**
	 * Get AO type
	 *
	 * @param integer $type type value of marche normal/cadre
	 *
	 * @return string
	 */
    public static function getMarcheType($type)
    {
    	global $langs;

    	switch ((int)$type) {
    		case 1:
    			return $langs->trans('MarcheNormal');
    			break;
    		case 2:
    			return $langs->trans('MarcheCadre');
    			break;
    	}
    }

    /**
	 * Get All AO types
	 *
	 * @return array
	 */
    public static function getMarcheTypes()
    {
    	global $langs;

    	return array(
    		1 => $langs->trans('MarcheNormal'),
    		2 => $langs->trans('MarcheCadre')
    	);
    }

    /**
     * Get status level : Drfat, Approved or Archived
     *
     * @param integer $statut value of status
     *
     * @return string
     */
    public function statusLevel($statut)
	{
		global $langs;

		switch ($statut) {
			case 0:
				return $langs->trans('Draft').' '.img_picto($langs->trans('Draft'),'statut0'); // Draft
				break;
			
			case 1:
				return $langs->trans('Approved').' '.img_picto($langs->trans('Approved'), 'statut6');
				break;

			case 2:
				return $langs->trans('Archived').' '.img_picto($langs->trans('Archived'),'statut5');
				break;
		}

		return false;
    }

    /**
     * Create general information about AO
     */
    public function getAppelOffreHeader()
    {
    	global $langs;
    	$html = '<table class="border" width="100%">';
		$html .= '<tr><td width="25%">'.$langs->trans("Subject").'</td>';
		$html .= '<td colspan="2">'. $this->subject .'</td></tr>';
		$html .= '<tr><td width="25%">'.$langs->trans("Label").'</td>';
		$html .= '<td colspan="2">'. $this->label .'</td></tr>';
		$html .= '<tr><td width="25%">'.$langs->trans("Ref").'</td>';
		$html .= '<td colspan="2">'. $this->ref .' &nbsp;&nbsp;&nbsp;&nbsp;'. $langs->trans("MonitoringDate").': '. dol_print_date($this->start_date, 'day') .'</td></tr>';
		$html .= '</table><br />';

		return $html;
    }
}
