#!/bin/bash
#
# COPYRIGHT:
# (c) 2001 Chris Arrowood (GNU LGPL V2.1)
# (c) 2017 Libre Expert
# You may view the full copyright text at:
# http://www.opensource.org/licenses/lgpl-license.html
#
# DESCRIPTION:
# A simple BASH script to do nightly backups to tarballs
# on a hard drive (not to tape)  Ideal for home linux users
# to easily backup thier system, provided they have an extra 
# hard drive.
#

function emailadmin {
    taskname="Dolibarr backup";
    out=`date +"%Y-%m-%d %H:%M:%S"`;
    host=$(/bin/hostname);
    
    if [ -f "/etc/synoinfo.conf" ]; 
    then
        source /etc/synoinfo.conf;
        headers=`printf "From: %s - %s <%s>\r\n" "$host" "$mailfrom" "$mailfrom"`;
    else
        headers=`printf "From: %s - %s <%s>\r\n" "$host" "admin@dolibarr" "admin@dolibarr"`;
    fi
    to="$1 <$1>"
    outcome=$2
    outcome=`printf "%s on %s was %s.\n\nThe database file size is %s bytes" "$taskname" "$host" "$outcome" $3`;
    
    if [ -f "/etc/synoinfo.conf" ];
    then
        subject=`printf "%s %s" "$eventsubjectprefix" "$outcome"`;
    else
        subject=`printf "%s" "$outcome"`;
    fi
    body=`printf "Dear user,\n\n%s.\n\nTask: %s\n\nSincerely,\nExtra Admin Modules for Dolibarr\n\n%s" "$outcome" "$taskname" "$out"`;
    
    if [ -f "/bin/php" ];
    then
        /bin/php -r "mail('$to', '$subject', '$body', '$headers');";
    else
        /usr/bin/php -r "mail('$to', '$subject', '$body', '$headers');";
    fi

    if [ -f "/usr/syno/bin/synodsmnotify" ];
    then
        /usr/syno/bin/synodsmnotify "@administrators" "System Event" "${outcome}";
    fi
}




###############################################
#              User Variables                 #
###############################################
#
# Modify these variables to suit your needs
#
SCRIPTDIR="$(dirname "$0")"
[ -f $SCRIPTDIR/backupconfig.conf ] && echo "Found configuration file" || { 
    echo "No configuration file found"
    exit 
}
source $SCRIPTDIR/backupconfig.conf

if [ $ACTIVATED -eq "0" ]; then
    echo "Backup not activated"
    exit
fi
echo "Backup activated"

HOUR=$(date +"%k")
if [ $HOUR -ne $BACKUPTIME ]; then 
    echo "Not time to backup"
    exit
fi
echo "Time to backup"

# Which day of the week do we want to do full backups? 0=Sunday
#  LEVEL0DAY=0
# Where to create the backups;
#  BACKUP_DIR=/mnt/backups/fwatl01
# Filesystems to backup seperated by spaces and the entire string in double quotes; each must start with /
#  FILESYSTEMS="/root /etc /var/lib/rpm /home /var/named"
# Should we email results? Also should we email critical errors?  0=false, 1=true 
  EMAIL=1
# EMAIL address to send results to
#  EMAILADDRESS=foo@bar.com
# Email Subject
  EMAILSUBJECT="$HOSTNAME Dolibarr Backup"
# Only keep last weeks level0 backup (0) or keep all lvl 0 backups (1).  Keeping all data may take a lot of space!
  KEEPALL=1
# Do we want to compress the backup file using gzip? 0=false, 1=true
  COMPRESS=1
# Should we compress the log file when we are done?  0=false, 1=true
  COMPRESSLOG=1
# If we are compressing, what level do we use?
  COMPRESSLEVEL=6
# Determines whether we see all output to screen. It will still go to log regardless of this value.   0=false, 1=true 
  QUIET=1
# Would you like to get detailed information from tar and gzip? 0=false, 1=true   
  VERBOSE=1


# DO NOT EDIT BELOW THIS LINE
#--------------------------------------------------------



###############################################
#     Application Variables -  DO NOT EDIT    #
###############################################
# Day of the week;
  DAYOFWEEK=`date +"%w"`
# DAYOFWEEK=3
# Folder for all daily backups
  DAILYBACKUPDIR=$BACKUP_DIR/DAILY
# Name of directory to create for current backup  
  TODAYSBACKUPDIR=$DAILYBACKUPDIR/$DAYOFWEEK
# directory to store last weeks data
  ARCHIVEDDATADIR=$BACKUP_DIR/ARCHIVED_BACKUPS
# Location of a file to hold the date stamp of last level 0 backup
  L0DATESTAMP=$BACKUP_DIR/.level0_datestamp
# Do I really need to explain this one ;-)
  NOW=`date +"%s"`
# Log dir
  LOGDIR=$BACKUP_DIR/LOGS
# Svript name
  SCRIPTNAME="Simple Bash Backup Script"
# Version
  VERSION=0.2
# Copyright
  COPYRIGHT="(c) 2001 Chris Arrowood, 2017 Libre Expert (GNU GPL V.2)"
# If user choose verbose, set the verbose command. Otherwise leave empty
if [ $VERBOSE -eq 1 ]
  then 
  VERBOSECOMMAND="--verbose"
fi
# Logfile
  LOGFILE=$LOGDIR/`date +"%Y%m%d_%H%M%S"`.log


######## Do some error checking #########
# Does backup dir exist?
if [ ! -d $BACKUP_DIR ]
  then
    echo "The specified backup directory $BACKUP_DIR does not exist. Creating it."
    mkdir -p $BACKUP_DIR
    if [ $EMAIL -eq 1 -a $? -ne 0 ]; then
        emailadmin $EMAILADDRESS "failure please check logs"
    fi
fi
# Does the daily backup dir exist? If not, create it.
if [ ! -d $DAILYBACKUPDIR ]
  then
    mkdir $DAILYBACKUPDIR
    if [ $EMAIL -eq 1 -a $? -ne 0 ]; then
        emailadmin $EMAILADDRESS "failure please check logs"
    fi
fi
 
# Does the daily backup dir exist? If not, create it.
if [ ! -d $LOGDIR ]
  then
    mkdir $LOGDIR
fi



####### Redirect Output to a logfile and screen - Couldnt get tee to work
exec 3>&1                         # create pipe (copy of stdout)
exec 1>$LOGFILE                   # direct stdout to file
exec 2>&1                         # uncomment if you want stderr too
if [ $QUIET -eq 0 ] 
  then tail --pid=$(pidof simplebashbu) -f $LOGFILE >&3 & # run tail in bg
fi



######## DO SOME PRINTING ###############
echo " "
echo "#######################################################################"
echo "$SCRIPTNAME "
echo "Version $VERSION" 
echo "$COPYRIGHT"
echo " "
echo "Host: $HOSTNAME "
echo "Start Time: $NOW"
echo "#######################################################################"
echo " "
echo " "
echo " "





######## Run Backup #########
#if day is LEVEL0DAY do full backup
if test $DAYOFWEEK -eq $LEVEL0DAY
  then
    LEVEL=0
    #we need to archive the last full backup to the weekold dir
    #make sure the week-old dir exists
    if test -d $ARCHIVEDDATADIR
      then 
        #remove old data unless KEEPALL is set to 1
        if test $KEEPALL -eq 0
          then rm -Rf $ARCHIVEDDATADIR/*
        fi
      else
        #the week-old data dir didnt exist; create it
	mkdir $ARCHIVEDDATADIR
        if [ $EMAIL -eq 1 -a $? -ne 0 ]; then
            emailadmin $EMAILADDRESS "failure please check logs"
        fi
        chmod 700 $ARCHIVEDDATADIR
    fi
    #move the last full backup to the weekold dir
    mv -f $TODAYSBACKUPDIR/* $ARCHIVEDDATADIR > /dev/null 2>&1
    #remove all daily backups since they are simply incrementals on the bu we just archived
    rm -Rf $DAILYBACKUPDIR/*
    #make todays dir since we just blew it away
    mkdir $TODAYSBACKUPDIR
    #do FULL backup for each filesystem specified 
    for BACKUPFILES in $FILESYSTEMS
    do
      #Create the filename; replace / with .
      WITHOUTSLASHES=`echo $BACKUPFILES | tr "/" "_"`
      WITHOUTLEADINGDOT=`echo $WITHOUTSLASHES | cut -b2-`
      OUTFILENAME=$WITHOUTLEADINGDOT-`date +"%Y%m%d_%H%M%S"`.tar
      OUTFILE=$TODAYSBACKUPDIR/$OUTFILENAME
      
      STARTTIME=`date`
      if [ $EXCLUDECONFPHP -eq 1 ]; then
        CONFFILE="htdocs/conf/conf.php"
        # Without the leading slash
        EXCLUDEFILE=`echo "$BACKUPFILES/$CONFFILE" | cut -b2-`
      else
        EXCLUDEFILE=""
      fi
      echo "EXCLUDEFILE="${EXCLUDEFILE//[$'\t\r\n']}
      echo " "
      echo " "
      echo "#######################################################################"
      echo "$STARTTIME: Creating a level $LEVEL backup of $BACKUPFILES to $OUTFILE" 
      echo "#######################################################################"
      #echo "tar --create" $VERBOSECOMMAND "--file" $OUTFILE "--label" ""Level-$LEVEL Backup from $LAST to $NOW"" $BACKUPFILES 

      tar --create $VERBOSECOMMAND \
        --file $OUTFILE \
        --label "Level-$LEVEL Backup ${NOW}" \
        $BACKUPFILES
      tar --file "$OUTFILE" --delete "$EXCLUDEFILE"
      if [ $EMAIL -eq 1 -a $? -ne 0 ]; then
        emailadmin $EMAILADDRESS "failure please check logs"
      fi
      if test $COMPRESS -eq 1
        then
        #gzip it
        gzip -$COMPRESSLEVEL $VERBOSECOMMAND $OUTFILE
        if [ $EMAIL -eq 1 -a $? -ne 0 ]; then
            emailadmin $EMAILADDRESS "failure please check logs"
        fi
        rm -f $OUTFILE
      fi
    done
    # Backup Dolibarr database
    RESULTFILE=$TODAYSBACKUPDIR/Dolibarr_database_`date +"%Y%m%d_%H%M%S"`.gz
    $MYSQLDUMP -u $DBUSER -p"$DBPASSWORD" $DBNAME | gzip > $RESULTFILE
    if [ $EMAIL -eq 1 -a $? -ne 0 ]; then
        emailadmin $EMAILADDRESS "failure please check logs"
    fi
    FILESIZE=$(stat -c%s "$RESULTFILE")

    # Does the timestamp file exist? If not, create it.
    if [ ! -w $L0DATESTAMP ]
      then
       touch $L0DATESTAMP
       if [ $EMAIL -eq 1 -a $? -ne 0 ]; then
         emailadmin $EMAILADDRESS "failure please check logs"
       fi
    fi    
    #record full backup timestamp
    echo $NOW > $L0DATESTAMP
else
    #we should do an incremental backup
    LEVEL=1
    # Does todays backup dir exist? If not, create it.
    if [ ! -d $TODAYSBACKUPDIR ]
      then
        mkdir $TODAYSBACKUPDIR
        if [ $EMAIL -eq 1 -a $? -ne 0 ]; then
            emailadmin $EMAILADDRESS "failure please check logs"
        fi
    fi    
    # Does the timestamp file exist? If not, create it.
    if [ ! -w $L0DATESTAMP ]
      then
       touch $L0DATESTAMP
       echo "0" > $L0DATESTAMP
    fi
    #get date of last full backup
    LAST=`cat $L0DATESTAMP`
    #Do a level 1 Backup for each filesystem specified
    for BACKUPFILES in $FILESYSTEMS
    do
      #Create the filename; replace / with .
      WITHOUTSLASHES=`echo $BACKUPFILES | tr "/" "_"`
      WITHOUTLEADINGDOT=`echo $WITHOUTSLASHES | cut -b2-`
      OUTFILENAME=$WITHOUTLEADINGDOT-`date +"%Y%m%d_%H%M%S"`.tar
      OUTFILE=$TODAYSBACKUPDIR/$OUTFILENAME
      STARTTIME=`date`
      if [ $EXCLUDECONFPHP -eq 1 ]; then
        CONFFILE="htdocs/conf/conf.php"
        # Without the leading slash
        EXCLUDEFILE=`echo "$BACKUPFILES/$CONFFILE" | cut -b2-`
      else
        EXCLUDEFILE=""
      fi
      echo " "
      echo " "
      echo "#######################################################################"
      echo "$STARTTIME: Creating a level $LEVEL backup of $BACKUPFILES to $OUTFILE" 
      echo "#######################################################################"
      #echo "tar --create" $VERBOSECOMMAND "--file" $OUTFILE "--after-date" "@$LAST" "--label" ""Level-$LEVEL Backup from $LAST to $NOW"" $BACKUPFILES 
      tar --create $VERBOSECOMMAND \
        --file $OUTFILE \
        --after-date "@$LAST" \
        --label "Level-$LEVEL Backup from $LAST to $NOW" \
        $BACKUPFILES
      tar --file $OUTFILE --delete "$EXCLUDEFILE"
      if [ $EMAIL -eq 1 -a $? -ne 0 ]; then
        emailadmin $EMAILADDRESS "failure please check logs"
      fi
      if test $COMPRESS -eq 1
        then
        #gzip it
        gzip -$COMPRESSLEVEL $VERBOSECOMMAND $OUTFILE
        if [ $EMAIL -eq 1 -a $? -ne 0 ]; then
            emailadmin $EMAILADDRESS "failure please check logs"
        fi
        rm -f $OUTFILE
      fi
    done
    # Backup Dolibarr database
    RESULTFILE=$TODAYSBACKUPDIR/Dolibarr_database_`date +"%Y%m%d_%H%M%S"`.sql.gz
    $MYSQLDUMP -u $DBUSER -p"$DBPASSWORD" $DBNAME | gzip > $RESULTFILE
    if [ $EMAIL -eq 1 -a $? -ne 0 ]; then
        emailadmin $EMAILADDRESS "failure please check logs"
    fi
    FILESIZE=$(stat -c%s "$RESULTFILE")
fi



SCRIPTFINISHTIME=`date`
echo " "
echo " "
echo " "
echo " "
echo "#######################################################################"
echo "Finish Time: $SCRIPTFINISHTIME"
echo " "
echo "  NOTE: Always examine the output of this script carefully for errors."
echo "        Also, be sure to verify your backups and your ability to "
echo "        retsore from them. "
echo "#######################################################################"



#email notification

if [ $EMAIL -eq 1 ]; then
  emailadmin $EMAILADDRESS "done" $FILESIZE
fi


#Fix stdout and stderr to go to console instead of og file since we are 
#about to compress the log file
exec 1>&3                   
exec 2>&3


#compress log file?
if test $COMPRESSLOG -eq 1
  then
  #gzip it
  gzip -$COMPRESSLEVEL $LOGFILE  > /dev/null 2>&1
  rm -f $LOGFILE  > /dev/null 2>&1
fi




exit 0
