<?php
/**
 * Fichier de classes pour gérer les hôtes
 *
 * Copyright (C) 2011-2019  Jean-François Ferry    <hello@librethic.io>
 * Copyright (C) 2011       Regis Houssin        <regis@dolibarr.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * @package Hosting
 * @author  jfefe    <hello@librethic.io>
 */

/**
 * Classe de gestion des hotes
 *
 * @class   Host
 * @package Hosting
 */
class Host extends CommonObject
{

    /* 0=>'Draft', 1=>'Online', '-1'=>'Offline', 9=>'Disabled' */
    const HOST_FK_STATUS_ONLINE =  1;
    const HOST_FK_STATUS_OFFLINE =  -1;
    const HOST_FK_STATUS_DRAFT =  0;
    const HOST_FK_STATUS_DISABLED =  9;

    /**
     * Object element
     *
     * @var string
     */
    public $element = 'hosting_host';

    /**
     * Name of table without prefix where object is stored
     *
     * @var string
     */
    public $table_element = 'host';

    /**
     * Does host support multicompany module ? 0=No test on entity, 1=Test with field entity, 2=Test with link by societe
     *
     * @var int
     */
    public $ismultientitymanaged = 0;

    /**
     * Does host support extrafields ? 0=No, 1=Yes
     *
     * @var int
     */
    public $isextrafieldmanaged = 1;

    /**
     * String with name of icon for host. Must be the part after the 'object_' into object_host.png
     *
     * @var string
     */
    public $picto = 'host@hosting';

    /**
     *  Array with all fields and their property. Do not use it as a static var. It may be modified by constructor.
     *
     *  'type' if the field format.
     *  'label' the translation key.
     *  'enabled' is a condition when the field must be managed.
     *  'visible' says if field is visible in list (Examples: 0=Not visible, 1=Visible on list and create/update/view forms, 2=Visible on list only. Using a negative value means field is not shown by default on list but can be selected for viewing)
     *  'notnull' is set to 1 if not null in database. Set to -1 if we must set data to null if empty ('' or 0).
     *  'index' if we want an index in database.
     *  'foreignkey'=>'tablename.field' if the field is a foreign key (it is recommanded to name the field fk_...).
     *  'position' is the sort order of field.
     *  'searchall' is 1 if we want to search in this field when making a search from the quick search button.
     *  'isameasure' must be set to 1 if you want to have a total on list for this field. Field type must be summable like integer or double(24,8).
     *  'help' is a string visible as a tooltip on field
     *  'comment' is not used. You can store here any text of your choice. It is not used by application.
     *  'default' is a default value for creation (can still be replaced by the global setup of default values)
     *  'showoncombobox' if field must be shown into the label of combobox
     *
     * @var array
     */
    public $fields=array(
        'rowid' => array('type'=>'integer', 'label'=>'TechnicalID', 'visible'=>-2, 'enabled'=>1, 'position'=>1, 'notnull'=>1, 'index'=>1, 'comment'=>"Id",),
        'ref' => array('type'=>'varchar(128)', 'label'=>'Ref', 'visible'=>-1, 'enabled'=>1, 'position'=>10, 'notnull'=>1, 'index'=>1, 'searchall'=>1, 'comment'=>"Reference of object",),
        'entity' => array('type'=>'integer', 'label'=>'Entity', 'visible'=>-2, 'enabled'=>1, 'position'=>20, 'notnull'=>1, 'index'=>1,),
        'label' => array('type'=>'varchar(255)', 'label'=>'HostName', 'visible'=>1, 'enabled'=>1, 'position'=>30, 'notnull'=>1, 'searchall'=>1),
        'fk_soc' => array('type'=>'integer:Societe:societe/class/societe.class.php', 'label'=>'ThirdParty', 'visible'=>1, 'enabled'=>1, 'position'=>50, 'notnull'=>1, 'index'=>1, 'searchall'=>1, 'help'=>"LinkToThirparty",),
        'description' => array('type'=>'html', 'label'=>'Description', 'visible'=>-1, 'enabled'=>1, 'position'=>60, 'notnull'=>-1,),
        'note_public' => array('type'=>'html', 'label'=>'NotePublic', 'visible'=>0, 'enabled'=>1, 'position'=>61, 'notnull'=>-1,),
        'note_private' => array('type'=>'html', 'label'=>'NotePrivate', 'visible'=>0, 'enabled'=>1, 'position'=>62, 'notnull'=>-1,),
        'date_creation' => array('type'=>'datetime', 'label'=>'DateCreation', 'visible'=>-2, 'enabled'=>1, 'position'=>500, 'notnull'=>1,),
        'tms' => array('type'=>'timestamp', 'label'=>'DateModification', 'visible'=>-2, 'enabled'=>1, 'position'=>501, 'notnull'=>1,),
        'fk_user_creat' => array('type'=>'integer:User:user/class/user.class.php', 'label'=>'UserAuthor', 'visible'=>-2, 'enabled'=>1, 'position'=>510, 'notnull'=>1,),
        'fk_user_modif' => array('type'=>'integer:User:user/class/user.class.php', 'label'=>'UserModif', 'visible'=>-2, 'enabled'=>1, 'position'=>511, 'notnull'=>-1,),
        'import_key' => array('type'=>'varchar(14)', 'label'=>'ImportId', 'visible'=>-2, 'enabled'=>1, 'position'=>1000, 'notnull'=>-1,),
        'status' => array('type'=>'integer', 'label'=>'Status', 'visible'=>1, 'enabled'=>1, 'position'=>1000, 'notnull'=>1, 'index'=>1, 'arrayofkeyval'=> array(0=>'Draft', 1=>'Online', '-1'=>'Offline', 9=>'Disabled')),
        'actif' => array('type'=>'boolean', 'label'=>'Active', 'visible'=>-2, 'enabled'=>1, 'position'=>10001, 'notnull'=>-1, 'index'=>1,),
        'category_code' => array('type'=>'sellist:c_host_category:label:code:', 'label'=>'HostCategoryFormLabel', 'visible'=>1, 'enabled'=>1, 'position'=>56, 'notnull'=>-1, 'index'=>1, 'comment'=>"Category code of dictionary",),
    );


    /**
     * Id of managed object
     *
     * @var integer
     */
    public $id;

    /**
     * Host company id
     *
     * @var integer
     */
    public $fk_soc;

    /**
     * Hostname (can be domain name)
     *
     * @var string
     */
    public $label;

    /**
     * To store various texte (public)
     *
     * @var string
     */
    public $note_public;
    
    /**
     * To store various texte (private)
     *
     * @var string
     */
    public $note_private;

    /**
     *  Host creation date
     *
     * @var string
     */
    public $date_creation = '';
    
    public $tms;
    public $fk_user_creat;
    public $fk_user_modif;
    public $import_key;
    public $status;
    

    /**
     * Active or not
     *
     * @var        integer
     * @deprecated use status instead
     */
    public $actif;

    /**
     * Category code for host (stored into dictionary)
     *
     * @var string
     */
    public $category_code;

    /**
     * Name of foreign key
     *
     * @var string
     */
    public $fk_element = 'fk_host';

    /**
     * Host object related (use for extended classes)
     *
     * @var Host
     */
    public $host;
    
    /**
     * Name of subtable line
     *
     * @var int
     */
    //public $table_element_line = 'hostdet';

    /**
     * Field with ID of parent key if this field has a parent
     *
     * @var int
     */
    //public $fk_element = 'fk_host';

    /**
     * Name of subtable class that manage subtable lines
     *
     * @var int
     */
    //public $class_element_line = 'Hostline';

    /**
     * Array of child tables (child tables to delete before deleting a record)
     *
     * @var array
     */
    //protected $childtables=array('hostdet');

    /**
     * Array of subtable lines
     *
     * @var HostLine[]
     */
    public $lines = array();

   
    /**
     *     Constructor of class
     *
     * @param DoliDB $db Database handler
     */
    public function __construct($db)
    {
        $this->db = $db;
        
        $this->statuts_short=array(0=>'Draft', 1=>'Online', -1 => 'Offline', 9=>'Disabled');
        $this->statuts=array(0=>'Draft', 1 => 'Online',  -1 => 'Offline', 9=>'Disabled');
    }

    /**
     * Create object into database
     *
     * @param User $user      User that creates
     * @param bool $notrigger false=launch triggers after, true=disable triggers
     *
     * @return int <0 if KO, Id of created object if OK
     */
    public function create(User $user, $notrigger = false)
    {
        return $this->createCommon($user, $notrigger);
    }
    
    
    /**
     * Clone and object into another one
     *
     * @param User $user   User that creates
     * @param int  $fromid Id of object to clone
     *
     * @return mixed New object created, <0 if KO
     */
    public function createFromClone(User $user, $fromid)
    {
        global $hookmanager, $langs;
        $error = 0;
        
        dol_syslog(__METHOD__, LOG_DEBUG);
        
        $object = new self($this->db);
        
        $this->db->begin();
        
        // Load source object
        $object->fetchCommon($fromid);
        // Reset some properties
        unset($object->id);
        unset($object->fk_user_creat);
        unset($object->import_key);
        
        // Clear fields
        $object->ref = "copy_of_".$object->ref;
        $object->title = $langs->trans("CopyOf")." ".$object->title;
        // ...
        
        // Create clone
        $object->context['createfromclone'] = 'createfromclone';
        $result = $object->createCommon($user);
        if ($result < 0) {
            $error++;
            $this->error = $object->error;
            $this->errors = $object->errors;
        }
        
        // End
        if (!$error) {
            $this->db->commit();
            return $object;
        } else {
            $this->db->rollback();
            return -1;
        }
    }
        

    /**
     * Check properties of third party are ok
     *
     * @return int 0 if OK, <0 if KO
     */
    public function verify()
    {
        $this->errors = array();
        $result = 0;
        $this->label = trim($this->label);

        if (!$this->fk_soc) {
            $this->errors[] = 'ErrorBadThirdPartyName';
            $result = -2;
        }
        if (!$this->label) {
            $this->errors[] = 'ErrorBadHostName';
            $result = -2;
        }
        return $result;
    }

    /**
     *     Load object in memory from database
     *
     * @param int    $id  id object
     * @param string $ref ref for object
     *
     * @return int         <0 if KO, >0 if OK
     */
    public function fetch($id, $ref = null)
    {
        $result = $this->fetchCommon($id, $ref);
        if ($result > 0 && ! empty($this->table_element_line)) {
            $this->fetchLines();
        }
         
        //$this->category_code = $obj->category_code;
        //$this->category_label = !empty($obj->category_code) ? $langs->trans('HostCategoryShort' . $obj->category_code) : '';

        $this->socid = $this->fk_soc; // backward compatibility
        //$this->fetch_thirdparty();
        return $result;
    }

    /**
     *  Load object hosting in $this->host from the database
     *
     * @param int $fk_host Id Host
     *
     * @return void
     */
    public function fetch_host($fk_host)
    {
        if (empty($fk_host)) {
            return 0;
        }

        $this->host = new Host($this->db);
        return $this->host->fetch($fk_host);
    }

    /**
     *  Load all objects in memory from database
     *
     * @param string $sortorder sort order
     * @param string $sortfield sort field
     * @param int    $limit     limit page
     * @param int    $offset    page
     * @param int    $active    display active // 0 : show inactive only, 1 : show active only, 2 : display both
     * @param array  $filter    filter output
     *
     * @return int              <0 if KO, >0 if OK
     */
    public function fetch_all($sortorder, $sortfield, $limit, $offset, $active = 1, $filter = '')
    {
        global $langs;

        $extrafields = new ExtraFields($this->db);

        // fetch optionals attributes and labels
        $extralabels = $extrafields->fetch_name_optionals_label($this->table_element);


        $sql = "SELECT";
        $sql .= " t.rowid,";
        $sql .= " t.fk_soc,";
        $sql .= " t.label,";
        // $sql .= " t.notes,";
        $sql .= " t.date_creation,";
        $sql .= " t.actif,";
        $sql .= " t.category_code,";
        $sql .= " s.nom";
        // Add fields for extrafields
        foreach ($extrafields->attribute_list as $key => $val) {
            $sql .= ",ef." . $key . ' as options_' . $key;
        }
        $sql .= " FROM " . MAIN_DB_PREFIX . "host as t";
        $sql .= " LEFT JOIN " . MAIN_DB_PREFIX . "societe as s ON s.rowid=t.fk_soc ";
        if (is_array($extrafields->attribute_label) && count($extrafields->attribute_label)) {
            $sql .= " LEFT JOIN " . MAIN_DB_PREFIX . "host_extrafields as ef ON ef.fk_object=t.rowid";
        }

        if ($active == 2) {
            $sql .= " WHERE t.actif <> '2'";
        } else {
            $sql .= " WHERE t.actif = '" . $active . "'";
        }

        $sql.= " AND t.entity IN (".getEntity('host').")";
        $sql .= " AND  s.rowid=t.fk_soc";
        //Manage filter
        if (!empty($filter)) {
            foreach ($filter as $key => $value) {
                if (strpos($key, 'date')) {
                    $sql .= ' AND ' . $key . ' = \'' . $this->db->idate($value) . '\'';
                } elseif ($key == 'fk_soc') {
                    $sql .= ' AND ' . $key . '=' . $value;
                } elseif (strpos($key, 'ef.') !== false) {
                    $sql .= ' AND '.$value;
                } else {
                    $sql .= ' AND ' . $key . ' LIKE \'%' . $value . '%\'';
                }
            }
        }
        $sql .= " ORDER BY $sortfield $sortorder ";
        if ($limit > 0) {
            $sql .= $this->db->plimit($limit + 1, $offset);
        }

        $resql = $this->db->query($sql);
        dol_syslog(get_class($this) . "::fetch_all sql=" . $sql, LOG_DEBUG);
        $resql = $this->db->query($sql);

        if ($resql) {
            $this->line = array();
            $num = $this->db->num_rows($resql);
            $i = 0;

            if ($num) {
                while ($i < $num) {
                    $obj = $this->db->fetch_object($resql);

                    $line = new HostLine();

                    $line->rowid = $obj->rowid;
                    $line->fk_soc = $obj->fk_soc;
                    $line->label = trim($obj->label);
                    $line->datec = $this->db->jdate($obj->datec);
                    $line->ref = $obj->ref;
                    $line->actif = $obj->actif;

                    $line->category_code = $obj->category_code;
                    $line->category_label = !empty($obj->category_code) ? $langs->trans('HostCategoryShort' . $obj->category_code) : '';

                    $line->notes = $obj->notes;

                    // Extra fields
                    if (is_array($extrafields->attribute_label) && count($extrafields->attribute_label)) {
                        foreach ($extrafields->attribute_label as $key => $val) {
                            $tmpkey = 'options_' . $key;
                            $line->{$tmpkey} = $obj->$tmpkey;
                        }
                    }

                    $this->line[$i] = $line;
                    $i++;
                }
            }
            $this->db->free($resql);
            return $num;
        } else {
            $this->error = "Error " . $this->db->lasterror();
            dol_syslog(get_class($this) . "::fetch_all " . $this->error, LOG_ERR);
            return -1;
        }
    }

    /**
     * Update object into database
     *
     * @param User $user      User that modifies
     * @param bool $notrigger false=launch triggers after, true=disable triggers
     *
     * @return int             <0 if KO, >0 if OK
     */
    public function update(User $user, $notrigger = false)
    {
        return $this->updateCommon($user, $notrigger);
    }
    

    /**
     *   Delete object in database
     *
     * @param User $user      User that delete
     * @param int  $notrigger 0=launch triggers after, 1=disable triggers
     *
     * @return int                <0 if KO, >0 if OK
     */
    public function delete(User $user, $notrigger = false)
    {
        global $conf, $langs;
        $error = 0;

        // Test if child exists
        $objectisused = $this->isObjectUsed();
        if (empty($objectisused)) {
            return $this->deleteCommon($user, $notrigger);
        } else {
            dol_syslog("Can't remove host with id " . $id . ". There is " . $objectisused . " childs", LOG_WARNING);
            $this->error = "ErrorRecordIsUsedCantDelete";
            return 0;
        }
    }

    /**
     *     Return select list of hosts
     *
     * @param string $selected   Id host preselected
     * @param string $htmlname   Field name in form
     * @param int    $show_empty 0=liste sans valeur nulle, 1=ajoute valeur inconnue
     * @param array  $exclude    Array list of groups id to exclude
     * @param int    $disabled   If select list must be disabled
     * @param array  $include    Array list of hosts id to include
     * @param string $enableonly Array list of hosts id to be enabled. All other must be disabled
     *
     * @return void
     */
    public function select_hosts($selected = '', $htmlname = 'hostid', $show_empty = 0, $exclude = '', $disabled = 0, $include = '', $enableonly = '')
    {
        global $conf;

        // Permettre l'exclusion de groupes
        if (is_array($exclude)) {
            $excludeGroups = implode("','", $exclude);
        }

        // Permettre l'inclusion de groupes
        if (is_array($include)) {
            $includeGroups = implode("','", $include);
        }

        $out = '';

        // On recherche les hosts
        $sql = "SELECT h.rowid, h.label as hostname ";
        $sql .= " FROM " . MAIN_DB_PREFIX . "host as h ";
        if ($enableonly) {
            $sql .= " WHERE h.actif ='1' OR h.status='" . Host::HOST_FK_STATUS_ONLINE . "'";
        }
        if (is_array($exclude) && $excludeGroups) {
            $sql .= " AND h.rowid NOT IN ('" . $excludeGroups . "')";
        }

        if (is_array($include) && $includeGroups) {
            $sql .= " AND h.rowid IN ('" . $includeGroups . "')";
        }

        $sql .= " ORDER BY h.label ASC";

        dol_syslog("Form::select_hosts sql=" . $sql);
        $resql = $this->db->query($sql);
        if ($resql) {
            $out .= '<select class="flat" id="'.$htmlname.'" name="' . $htmlname . '"' . ($disabled ? ' disabled="true"' : '') . '>';
            if ($show_empty) {
                $out .= '<option value="-1"' . ($id == -1 ? ' selected="selected"' : '') . '>&nbsp;</option>' . "\n";
            }

            $num = $this->db->num_rows($resql);
            $i = 0;
            if ($num) {
                while ($i < $num) {
                    $obj = $this->db->fetch_object($resql);
                    $disableline = 0;
                    if (is_array($enableonly) && sizeof($enableonly) && !in_array($obj->rowid, $enableonly)) {
                        $disableline = 1;
                    }

                    $out .= '<option value="' . $obj->rowid . '"';
                    if ($disableline) {
                        $out .= ' disabled="true"';
                    }

                    if ((is_object($selected) && $selected->id == $obj->rowid) || (!is_object($selected) && $selected == $obj->rowid)) {
                        $out .= ' selected="selected"';
                    }
                    $out .= '>';

                    $out .= $obj->hostname;

                    $out .= '</option>';
                    $i++;
                }
            }
            $out .= '</select>';
        } else {
            dol_print_error($this->db);
        }

        return $out;
    }

    /**
     *  Return an array of host type
     *
     * @return array Managed host type
     */
    public function getListOfManagedType()
    {
        global $conf, $langs;

        $ret = array();

        $sql = "SELECT rowid, code, label, description";
        $sql .= " FROM " . MAIN_DB_PREFIX . "c_host_type";
        $sql .= $this->db->order("code, label");
        $resql = $this->db->query($sql);
        if ($resql) {
            $num = $this->db->num_rows($resql);
            $i = 0;
            while ($i < $num) {
                $obj = $this->db->fetch_object($resql);

                $ret[] = array(
                    'rowid' => $obj->rowid,
                    'code' => $obj->code,
                    'label' => $obj->label,
                    'description' => $obj->description,
                );
                $i++;
            }
        } else {
            dol_print_error($this->db);
        }

        return $ret;
    }

    /**
     * Return id of host for an extension
     *
     * @param string $element    Name of element
     * @param int    $element_id Id of element which retrieve host
     *
     * @return int
     */
    public function getHostIdForDef($element, $element_id)
    {
        $sql_type = "SELECT fk_host FROM " . MAIN_DB_PREFIX . "host_def";
        $sql_type .= " WHERE element='" . strtoupper($element) . "'";
        $sql_type .= " AND element_id='" . strtoupper($element_id) . "'";

        dol_syslog("getHostIdForDef : $element-$element_id");

        if ($resql = $this->db->query($sql_type)) {
            if ($this->db->num_rows($resql)) {
                $obj = $this->db->fetch_object($resql);
                return $obj->fk_host;
            }
        } else {
            dol_print_error($db);
        }
    }

    /**
     *  Return a link to the object card (with optionaly the picto)
     *
     * @param int    $withpicto             Include picto in link (0=No picto, 1=Include picto into link, 2=Only picto)
     * @param string $option                On what the link point to ('nolink', ...)
     * @param int    $notooltip             1=Disable tooltip
     * @param string $morecss               Add more css on link
     * @param int    $save_lastsearch_value -1=Auto, 0=No save of lastsearch_values when clicking, 1=Save lastsearch_values whenclicking
     * @param int    $addlabel              0=Default, 1=Add label into string, >1=Add first chars into string
     *
     * @return string String with URL
     */
    public function getNomUrl($withpicto = 0, $option = '', $notooltip = 0, $morecss = '', $save_lastsearch_value = -1, $addlabel = 0)
    {
        global $db, $conf, $langs;
        global $dolibarr_main_authentication, $dolibarr_main_demo;
        global $menumanager;
        
        if (! empty($conf->dol_no_mouse_hover)) {
            $notooltip = 1;   // Force disable tooltips
        }
        
        $result = '';
        $companylink = '';
        
        $label = '<u>' . $langs->trans("Host") . '</u>';
        $label.= '<br>';
        $label.= '<b>' . $langs->trans('Ref') . ':</b> ' . $this->ref;
        $label.= '<br>';
        $label.= '<b>' . $langs->trans('HostName') . ':</b> ' . $this->label;
        
        $url = dol_buildpath('/hosting/host_card.php', 1).'?id='.$this->id;
        
        if ($option != 'nolink') {
            if (preg_match('/\.php$/', $option)) {
                $url = dol_buildpath($option, 1) . '?id=' . $this->id;
            } elseif ($option == 'monitor') {
                $url = dol_buildpath('/monitoring/monitor_list.php', 1).'?hostid=' . $this->id;
            }
 
            // Add param to save lastsearch_values or not
            $add_save_lastsearch_values=($save_lastsearch_value == 1 ? 1 : 0);
            if ($save_lastsearch_value == -1 && preg_match('/list\.php/', $_SERVER["PHP_SELF"])) {
                $add_save_lastsearch_values=1;
            }
            if ($add_save_lastsearch_values) {
                $url.='&save_lastsearch_values=1';
            }
        }
        
        $linkclose='';
        if (empty($notooltip)) {
            if (! empty($conf->global->MAIN_OPTIMIZEFORTEXTBROWSER)) {
                $label=$langs->trans("ShowHost");
                $linkclose.=' alt="'.dol_escape_htmltag($label, 1).'"';
            }
            $linkclose.=' title="'.dol_escape_htmltag($label, 1).'"';
            $linkclose.=' class="classfortooltip'.($morecss?' '.$morecss:'').'"';

            if (! is_object($hookmanager)) {
                include_once DOL_DOCUMENT_ROOT.'/core/class/hookmanager.class.php';
                $hookmanager=new HookManager($this->db);
            }
            $hookmanager->initHooks(array('hostdao'));
            $parameters=array('id'=>$this->id);
            // Note that $action and $object may have been modified by some hooks
            $reshook=$hookmanager->executeHooks('getnomurltooltip', $parameters, $this, $action);
            if ($reshook > 0) {
                $linkclose = $hookmanager->resPrint;
            }
        } else {
            $linkclose = ($morecss?' class="'.$morecss.'"':'');
        }
        
        $picto = $this->picto;

        $linkstart = '<a href="'.$url.'"';
        $linkstart.=$linkclose.'>';
        $linkend='</a>';
        
        $result .= $linkstart;
        if ($withpicto) {
            $result.=img_object(($notooltip?'':$label), ($picto ? $picto : 'generic'), ($notooltip?(($withpicto != 2) ? 'class="paddingright"' : ''):'class="'.(($withpicto != 2) ? 'paddingright ' : '').'classfortooltip"'), 0, 0, $notooltip?0:1);
        }
        if ($withpicto != 2) {
             $result.= $this->ref;
        }
        $result .= $linkend;
        $sep = ' ';
        if ($withpicto != 2) {
            $result.=(($addlabel && $this->label) ? $sep . dol_trunc($this->label, ($addlabel > 1 ? $addlabel : 0)) : '');
        }
        
        return $result;
    }
    
    /**
     *  Retourne le libelle du status d'un user (actif, inactif)
     *
     * @param int $mode 0=libelle long, 1=libelle court, 2=Picto + Libelle court, 3=Picto, 4=Picto + Libelle long, 5=Libelle court + Picto
     *
     * @return string Label of status
     */
    public function getLibStatut($mode = 0)
    {
        return $this->LibStatut($this->status, $mode);
    }
    
    /**
     *  Return the status
     *
     * @param int $status Id status
     * @param int $mode   0=long label, 1=short label, 2=Picto + short label, 3=Picto, 4=Picto + long label, 5=Short label + Picto, 6=Long label + Picto
     *
     * @return string Label of status
     */
    public static function LibStatut($status, $mode = 0)
    {
        global $langs;
        if ($mode == 0) {
            $prefix='';
            if ($status == 0) {
                return $langs->trans('Draft');
            }
            if ($status == 1) {
                return $langs->trans('Online');
            }
            if ($status == -1) {
                return $langs->trans('Offline');
            }
            if ($status == 9) {
                return $langs->trans('Disabled');
            }
        }
        if ($mode == 1) {
            if ($status == 0) {
                return $langs->trans('Draft');
            }
            if ($status == 1) {
                return $langs->trans('Online');
            }
            if ($status == 1) {
                return $langs->trans('Online');
            }
            if ($status == -1) {
                return $langs->trans('Offline');
            }
            if ($status == 9) {
                return $langs->trans('Disabled');
            }
        }
        if ($mode == 2) {
            if ($status == 0) {
                return img_picto($langs->trans('Draft'), 'statut5').' '.$langs->trans('Disabled');
            }
            if ($status == 1) {
                return img_picto($langs->trans('Online'), 'statut4').' '.$langs->trans('Online');
            }
            if ($status == -1) {
                return img_picto($langs->trans('Offline'), 'statut7').' '.$langs->trans('Offline');
            }
            if ($status == 9) {
                return img_picto($langs->trans('Disabled'), 'statut5').' '.$langs->trans('Disabled');
            }
        }
        if ($mode == 3) {
            if ($status == 0) {
                return img_picto($langs->trans('Draft'), 'statut5');
            }
            if ($status == 1) {
                return img_picto($langs->trans('Online'), 'statut4');
            }
            if ($status == -1) {
                return img_picto($langs->trans('Offline'), 'statut7');
            }
            if ($status == 9) {
                return img_picto($langs->trans('Disabled'), 'statut5');
            }
        }
        if ($mode == 4) {
            if ($status == 0) {
                return img_picto($langs->trans('Draft'), 'statut5').' '.$langs->trans('Draft');
            }
            if ($status == 1) {
                return img_picto($langs->trans('Online'), 'statut4').' '.$langs->trans('Online');
            }
            if ($status == -1) {
                return img_picto($langs->trans('Offline'), 'statut7').' '.$langs->trans('Offline');
            }
            if ($status == 9) {
                return img_picto($langs->trans('Disabled'), 'statut5').' '.$langs->trans('Disabled');
            }
        }
        if ($mode == 5) {
            if ($status == 0) {
                return $langs->trans('Draft').' '.img_picto($langs->trans('Disabled'), 'statut5');
            }
            if ($status == -1) {
                return $langs->trans('Offline').' '.img_picto($langs->trans('Offline'), 'statut7');
            }
            if ($status == 1) {
                return $langs->trans('Online').' '.img_picto($langs->trans('Online'), 'statut4');
            }
            if ($status == 9) {
                return $langs->trans('Disabled').' '.img_picto($langs->trans('Disabled'), 'statut5');
            }
        }
        if ($mode == 6) {
            if ($status == 0) {
                return $langs->trans('Draft').' '.img_picto($langs->trans('Draft'), 'statut0');
            }
            if ($status == -1) {
                return $langs->trans('Offline').' '.img_picto($langs->trans('Offline'), 'statut7');
            }
            if ($status == 1) {
                return $langs->trans('Online').' '.img_picto($langs->trans('Online'), 'statut4');
            }
            if ($status == 9) {
                return $langs->trans('Disabled').' '.img_picto($langs->trans('Disabled'), 'statut5');
            }
        }
    }

    /**
     * Renvoie image selon l'état de l'hote (champ 'actif')
     *
     * @return string            Chaine avec URL
     */
    public function getImgState()
    {
        global $langs;

        $result = '';

        $img = ($this->actif ? 'on' : 'off');
        $alt = ($this->actif ? $langs->trans('Activated') : $langs->trans('Disabled'));
        return img_picto($alt, $img);
    }

    /**
     * Charge les informations d'ordre info dans l'objet host
     *
     * @param int $id Id of host
     *
     * @return void
     */
    public function info($id)
    {
        $sql = 'SELECT rowid, date_creation as datec, tms as datem,';
        $sql.= ' fk_user_creat, fk_user_modif';
        $sql.= ' FROM '.MAIN_DB_PREFIX.$this->table_element.' as t';
        $sql.= ' WHERE t.rowid = '.$id;
        $result=$this->db->query($sql);
        if ($result) {
            if ($this->db->num_rows($result)) {
                $obj = $this->db->fetch_object($result);
                $this->id = $obj->rowid;
                if ($obj->fk_user_author) {
                    $cuser = new User($this->db);
                    $cuser->fetch($obj->fk_user_author);
                    $this->user_creation   = $cuser;
                }
                
                if ($obj->fk_user_valid) {
                    $vuser = new User($this->db);
                    $vuser->fetch($obj->fk_user_valid);
                    $this->user_validation = $vuser;
                }
                
                if ($obj->fk_user_cloture) {
                    $cluser = new User($this->db);
                    $cluser->fetch($obj->fk_user_cloture);
                    $this->user_cloture   = $cluser;
                }
                
                $this->date_creation     = $this->db->jdate($obj->datec);
                $this->date_modification = $this->db->jdate($obj->datem);
                $this->date_validation   = $this->db->jdate($obj->datev);
            }
            
            $this->db->free($result);
        } else {
            dol_print_error($this->db);
        }
    }
    
    
    
    /**
     * Initialise object with example values
     * Id must be 0 if object instance is a specimen
     *
     * @return void
     */
    public function initAsSpecimen()
    {
        $this->initAsSpecimenCommon();
    }
    
    
    /**
     * Action executed by scheduler
     * CAN BE A CRON TASK
     *
     * @return int 0 if OK, <>0 if KO (this function is used also by cron so only 0 is OK)
     */
    public function doScheduledJob()
    {
        global $conf, $langs;
        
        $this->output = '';
        $this->error='';
        
        dol_syslog(__METHOD__, LOG_DEBUG);
        
        // ...
        
        return 0;
    }
    
    /**
     * Function to check if an object is used by others.
     * Check is done into llx_host_type table. There is no check into llx_element_element.
     *
     * @param int $id Force id of object
     *
     * @return int <0 if KO, 0 if not used, >0 if already used
     */
    public function isObjectUsed($id = 0)
    {
        if (empty($id)) {
            $id = $this->id;
        }
        
        $haschild = 0;

        // Check if host can be deleted
        $nb = 0;
        $sql = "SELECT COUNT(*) as nb from " . MAIN_DB_PREFIX . 'host_def';
        $sql .= " WHERE " . $this->fk_element . " = " . $id;

        $resql = $this->db->query($sql);
        if ($resql) {
            $obj = $this->db->fetch_object($resql);
            $haschild += $obj->nb;
        } else {
            $this->error = $this->db->lasterror();
            dol_syslog(get_class($this) . "::isObjectUsed error -1 " . $this->error, LOG_ERR);
            return -1;
        }

        if ($haschild > 0) {
            $this->error = "ErrorRecordHasChildren";
            return $haschild;
        } else {
            return 0;
        }
    }
    
    /**
     * Show a table with small host datas
     *
     * @param User   $user Object user who view host
     * @param string $tab  Tab name
     *
     * @return void
     */
    public function showHostCardInfos($user, $tab = 'tabMonitoring')
    {
        global $langs;

        $head = hosting_prepare_head($this);
        dol_fiche_head($head, $tab, $langs->trans("Host"), 0, 'host@hosting');

        $linkback = '<a href="' .dol_buildpath('/hosting/host_list.php', 1) . '?restore_lastsearch_values=1' . (! empty($socid) ? '&socid=' . $socid : '') . '">' . $langs->trans("BackToList") . '</a>';
        $morehtmlref='<div class="refidno">';
        $morehtmlref.=$this->label;
        // Thirdparty
        if ($this->fetch_thirdparty('') && $this->thirdparty->id) {
            $morehtmlref.='<br>'.$langs->trans('ThirdParty') . ' : ' . $this->thirdparty->getNomUrl(1);
        }
        $morehtmlref.='<br />'.$langs->trans('DateCreation').' : '.dol_print_date($this->date_creation);
        $morehtmlref.='</div>';
        dol_banner_tab($this, 'ref', $linkback, 1, 'ref', 'ref', $morehtmlref);

        print dol_fiche_end();
    }

    /**
     * Charge dans cache la liste des catégories d'hôtes (paramétrable dans dictionnaire)
     *
     * @return int Nb lignes chargees, 0 si deja chargees, <0 si ko
     */
    public function load_cache_categories_hosts()
    {
        global $langs;

        if (count($this->cache_categories_hosts)) {
            return 0;
        }
        // Cache deja charge

        $sql = "SELECT rowid, code, label, use_default, pos, description";
        $sql .= " FROM " . MAIN_DB_PREFIX . "c_host_category";
        $sql .= " WHERE active > 0";
        $sql .= " ORDER BY pos ASC";
        dol_syslog(get_class($this) . "::load_cache_categories_hosts sql=" . $sql, LOG_DEBUG);
        $resql = $this->db->query($sql);
        if ($resql) {
            $num = $this->db->num_rows($resql);
            $i = 0;
            while ($i < $num) {
                $obj = $this->db->fetch_object($resql);
                // Si traduction existe, on l'utilise, sinon on prend le libelle par defaut
                $label = ($langs->trans("TicketTypeShort" . $obj->code) != ("HostCategoryShort" . $obj->code) ? $langs->trans("HostCategoryShort" . $obj->code) : ($obj->label != '-' ? $obj->label : ''));
                $this->cache_categories_hosts[$obj->rowid]['code'] = $obj->code;
                $this->cache_categories_hosts[$obj->rowid]['label'] = $label;
                $this->cache_categories_hosts[$obj->rowid]['use_default'] = $obj->use_default;
                $this->cache_categories_hosts[$obj->rowid]['pos'] = $obj->pos;
                $i++;
            }
            return $num;
        } else {
            dol_print_error($this->db);
            return -1;
        }
    }

    /**
     * Return html list of host categories
     *
     * @param string $selected    Category code pre-selectionnée
     * @param string $htmlname    Nom de la zone select
     * @param string $filtertype  To filter on field type in llx_c_ticketsup_category (array('code'=>xx,'label'=>zz))
     * @param int    $format      0=id+libelle, 1=code+code, 2=code+libelle, 3=id+code
     * @param int    $empty       1=peut etre vide, 0 sinon
     * @param int    $noadmininfo 0=Add admin info, 1=Disable admin info
     * @param int    $maxlength   Max length of label
     *
     * @return void
     */
    public function select_categories_hosts($selected = '', $htmlname = 'category_code', $filtertype = '', $format = 0, $empty = 0, $noadmininfo = 0, $maxlength = 0)
    {
        global $langs, $user;

        dol_syslog(get_class($this) . "::select_categories_hosts " . $selected . ", " . $htmlname . ", " . $filtertype . ", " . $format, LOG_DEBUG);

        $filterarray = array();

        if ($filtertype != '' && $filtertype != '-1') {
            $filterarray = explode(',', $filtertype);
        }

        $this->load_cache_categories_hosts();

        print '<select id="select' . $htmlname . '" class="flat select_ticketcategory" name="' . $htmlname . '">';
        if ($empty) {
            print '<option value="">&nbsp;</option>';
        }

        if (is_array($this->cache_categories_hosts) && count($this->cache_categories_hosts)) {
            foreach ($this->cache_categories_hosts as $id => $arraycategories) {
                // On passe si on a demande de filtrer sur des modes de paiments particuliers
                if (count($filterarray) && !in_array($arraycategories['type'], $filterarray)) {
                    continue;
                }

                // We discard empty line if showempty is on because an empty line has already been output.
                if ($empty && empty($arraycategories['code'])) {
                    continue;
                }

                if ($format == 0) {
                    print '<option value="' . $id . '"';
                }

                if ($format == 1) {
                    print '<option value="' . $arraycategories['code'] . '"';
                }

                if ($format == 2) {
                    print '<option value="' . $arraycategories['code'] . '"';
                }

                if ($format == 3) {
                    print '<option value="' . $id . '"';
                }

                // Si selected est text, on compare avec code, sinon avec id
                if (preg_match('/[a-z]/i', $selected) && $selected == $arraycategories['code']) {
                    print ' selected="selected"';
                } elseif ($selected == $id) {
                    print ' selected="selected"';
                } elseif ($arraycategories['use_default'] == "1") {
                    print ' selected="selected"';
                }

                print '>';

                if ($format == 0) {
                    $value = ($maxlength ? dol_trunc($arraycategories['label'], $maxlength) : $arraycategories['label']);
                }

                if ($format == 1) {
                    $value = $arraycategories['code'];
                }

                if ($format == 2) {
                    $value = ($maxlength ? dol_trunc($arraycategories['label'], $maxlength) : $arraycategories['label']);
                }

                if ($format == 3) {
                    $value = $arraycategories['code'];
                }

                print $value ? $value : '&nbsp;';
                print '</option>';
            }
        }
        print '</select>';
        if ($user->admin && !$noadmininfo) {
            print info_admin($langs->trans("YouCanChangeValuesForThisListFromDictionnarySetup"), 1);
        }
    }

    /**
     * Get a default reference
     *
     * @param Societe $thirdparty Thirdparty object
     *
     * @global type $conf
     * @return string   Reference
     */
    public function getDefaultRef($thirdparty = '')
    {
        global $conf;
        
        $defaultref = '';
        $modele = empty($conf->global->HOSTING_ADDON) ? 'mod_hosting_simple' : $conf->global->HOSTING_ADDON;
        
        // Search template files
        $file = '';
        $classname = '';
        $filefound = 0;
        $dirmodels = array_merge(array('/'), (array) $conf->modules_parts['models']);
        foreach ($dirmodels as $reldir) {
            $file = dol_buildpath($reldir . "core/modules/hosting/" . $modele . '.php', 0);
            if (file_exists($file)) {
                $filefound = 1;
                $classname = $modele;
                break;
            }
        }
        
        if ($filefound) {
            $result = dol_include_once($reldir . "core/modules/hosting/" . $modele . '.php');
            $modHosting = new $classname;
            
            $defaultref = $modHosting->getNextValue($thirdparty, $this);
        }
        
        if (is_numeric($defaultref) && $defaultref <= 0) {
            $defaultref = '';
        }
        
        return $defaultref;
    }
}

/**
 *     Assign an hosting extension to an host
 *
 * @param int    $hostid     Host id
 * @param int    $socid      Company ID
 * @param string $element    Linked object element
 * @param int    $element_id Linked object element ID
 *
 * @return int    Linked object element ID
 */
function AssignHostExtension($hostid, $socid, $element, $element_id)
{
    global $db;

    $sql_type = "SELECT rowid FROM " . MAIN_DB_PREFIX . "c_host_type";
    $sql_type .= " WHERE code='" . strtoupper($element) . "'";
    if ($resql = $db->query($sql_type)) {
        if ($db->num_rows($resql)) {
            $obj = $db->fetch_object($resql);
            $typeid = $obj->rowid;

            //$sql = "DELETE FROM ".MAIN_DB_PREFIX."host_def";
            //$sql .= " WHERE fk_soc=".$socid." AND fk_host=".$hostid." AND fk_type=".$typeid;

            $sql = "INSERT INTO " . MAIN_DB_PREFIX . "host_def (datec,fk_soc, fk_host, fk_type, element, element_id)";
            $sql .= " VALUES ('" . $db->idate(mktime()) . "'," . $socid . "," . $hostid . "," . $typeid . ",'" . strtoupper($element) . "', $element_id)";

            if ($db->query($sql)) {
                return $element_id;
            } else {
                dol_print_error($db);
            }
        }
    } else {
        dol_print_error($db);
    }
}

/**
 * Unassign an hosting extension from an host
 *
 * @param int    $hostid     Host id
 * @param int    $socid      Company ID
 * @param string $element    Linked object element
 * @param int    $element_id ID for element
 *
 * @return int        $element_id        inked object element ID
 */
function UnAssignHostExtension($hostid, $socid, $element, $element_id)
{
    global $db;

    $sql_type = "SELECT rowid FROM " . MAIN_DB_PREFIX . "c_host_type";
    $sql_type .= " WHERE code='" . strtoupper($element) . "'";
    if ($resql = $db->query($sql_type)) {
        if ($db->num_rows($resql)) {
            $obj = $db->fetch_object($resql);
            $typeid = $obj->rowid;

            $sql = "DELETE FROM " . MAIN_DB_PREFIX . "host_def";
            $sql .= " WHERE fk_soc=" . $socid . " AND fk_host='" . $hostid . "' AND fk_type='" . $typeid . "' AND element_id='" . $element_id . "'";

            dol_syslog("UnAssignHostExtension for host $hostid : $element-$element_id");
            if ($db->query($sql)) {
                return 1;
            } else {
                dol_print_error($db);
                return -1;
            }
        }
    } else {
        dol_print_error($db);
        return -1;
    }

    return -1;
}

/**
 * Host line class
 *
 * @package Hosting
 */
class HostLine
{

    /**
     * To store db handler
     *
     * @var Object    To store db handler
     */
    public $db;

    /**
     * To return error code (or message)
     *
     * @var string
     */
    public $error;

    /**
     * To return several error codes (or messages)
     *
     * @var array
     */
    public $errors = array();

    /**
     * Object element
     *
     * @var string
     */
    public $element = 'hosting_host';

    /**
     * Name of table without prefix where object is stored
     *
     * @var string
     */
    public $table_element = 'host';

    /**
     * Id of managed object
     *
     * @var integer
     */
    public $rowid;

    /**
     * Host company id
     *
     * @var integer
     */
    public $socid;

    /**
     * Hostname (can be domain name)
     *
     * @var string
     */
    public $label;

    /**
     * To store various texte (public)
     *
     * @var string
     */
    public $notes;

    /**
     * Host creation date
     *
     * @var string
     */
    public $datec = '';

    /**
     * Active or not (not used)
     *
     * @var integer
     */
    public $actif;

    /**
     * Constructor
     */
    public function __construct()
    {
        return 1;
    }
}
