<?php
/* Copyright (C) 2012-2017		Charlene Benke	<charlie@patas-monkey.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 * or see http://www.gnu.org/
 */

/**
 *	\file	   extraodt/core/modules/product/doc/pdf_malta.modules.php
 *	\ingroup	equipement
 *	\brief	  Fichier de une édition produit avec les infos additionnels
 */

require_once DOL_DOCUMENT_ROOT."/product/class/product.class.php";
require_once DOL_DOCUMENT_ROOT."/compta/facture/class/facture.class.php";
require_once DOL_DOCUMENT_ROOT."/fourn/class/fournisseur.facture.class.php";
require_once DOL_DOCUMENT_ROOT."/product/stock/class/entrepot.class.php";

require_once DOL_DOCUMENT_ROOT.'/core/modules/product/modules_product.php';
require_once DOL_DOCUMENT_ROOT."/core/lib/company.lib.php";
require_once DOL_DOCUMENT_ROOT.'/core/lib/pdf.lib.php';
require_once DOL_DOCUMENT_ROOT.'/core/lib/date.lib.php';


/**
 *	Class to build equipement documents with model Soleil
 */
class pdf_malta extends ModeleProduct
{
	var $db;
	var $name;
	var $description;
	var $type;

	var $phpmin = array(4,3,0); // Minimum version of PHP required by module
	var $version = '4.0.+2.0.2';

	var $page_largeur;
	var $page_hauteur;
	var $format;
	var $marge_gauche;
	var	$marge_droite;
	var	$marge_haute;
	var	$marge_basse;

	/**
	 *	Constructor
	 *
	 *  @param		DoliDB		$db	  Database handler
	 */
	function __construct($db)
	{
		global $conf, $langs, $mysoc;

		$this->db = $db;
		$this->name = 'malta';
		$this->description = $langs->trans("DocumentProductModelStandard");

		// Dimension page pour format A4
		$this->type = 'pdf';
		$formatarray=pdf_getFormat();
		$this->page_largeur = $formatarray['width'];
		$this->page_hauteur = $formatarray['height'];
		$this->format = array($this->page_largeur, $this->page_hauteur);
		$this->marge_gauche=10;
		$this->marge_droite=10;
		$this->marge_haute=10;
		$this->marge_basse=10;

		$this->option_logo = 1;					// Affiche logo
		$this->option_tva = 0;					 // Gere option tva FACTURE_TVAOPTION
		$this->option_modereg = 0;				 // Affiche mode reglement
		$this->option_condreg = 0;				 // Affiche conditions reglement
		$this->option_codeproduitservice = 0;	  // Affiche code produit-service
		$this->option_multilang = 0;			   // Dispo en plusieurs langues
		$this->option_draft_watermark = 1;		   //Support add of a watermark on drafts

		// Recupere emmetteur
		$this->emetteur=$mysoc;
		if (! $this->emetteur->code_pays) 
			$this->emetteur->code_pays=substr($langs->defaultlang, -2);	// By default, if not defined

		// Defini position des colonnes
		$this->posxdesc=$this->marge_gauche+1;
	}

	/**
	 *  Function to build pdf onto disk
	 *
	 *  @param		object	$object				Object to generate
	 *  @param		object	$outputlangs		Lang output object
	 *  @return		int							1=ok, 0=ko
	 */
	function write_file($object, $outputlangs)
	{
		global $user, $langs, $conf, $mysoc;
		$default_font_size = pdf_getPDFFontSize($outputlangs);

		if (! is_object($outputlangs)) 
			$outputlangs=$langs;
		// For backward compatibility with FPDF, force output charset to ISO, because FPDF expect text to be encoded in ISO
		if (! empty($conf->global->MAIN_USE_FPDF)) 
			$outputlangs->charset_output='ISO-8859-1';

		$outputlangs->load("main");
		$outputlangs->load("dict");
		$outputlangs->load("products");
		$outputlangs->load("extraodt@extraodt");
		
		if ($conf->global->MAIN_MODULE_EQUIPEMENT)
			$outputlangs->load("equipement@equipement");
		if ($conf->global->MAIN_MODULE_FACTORY)
			$outputlangs->load("factory@factory");

		if ($conf->product->dir_output) {
			$object->fetch_thirdparty();

			$objectref = dol_sanitizeFileName($object->ref);
			$dir = $conf->product->dir_output;
			if (! preg_match('/specimen/i', $objectref))
				$dir.= "/" . $objectref;
			$file = $dir."/".$objectref.".pdf";

			if (! file_exists($dir)) {
				if (dol_mkdir($dir) < 0) {
					$this->error=$outputlangs->trans("ErrorCanNotCreateDir", $dir);
					return 0;
				}
			}

			if (file_exists($dir)) {
				$pdf=pdf_getInstance($this->format);

				if (class_exists('TCPDF')) {
					$pdf->setPrintHeader(false);
					$pdf->setPrintFooter(false);
				}
				$pdf->SetFont(pdf_getPDFFont($outputlangs));
				// Set path to the background PDF File
				if (empty($conf->global->MAIN_DISABLE_FPDI) && ! empty($conf->global->MAIN_ADD_PDF_BACKGROUND)) {
					$pagecount = $pdf->setSourceFile(
						$conf->mycompany->dir_output.'/'.$conf->global->MAIN_ADD_PDF_BACKGROUND
					);
					$tplidx = $pdf->importPage(1);
				}

				$pdf->Open();
				$pagenb=0;
				$pdf->SetDrawColor(128, 128, 128);

				$pdf->SetTitle($outputlangs->convToOutputCharset($object->ref));
				$pdf->SetSubject($outputlangs->transnoentities("ProductCard"));
				$pdf->SetCreator("Dolibarr ".DOL_VERSION);
				$pdf->SetAuthor($outputlangs->convToOutputCharset($user->getFullName($outputlangs)));
				$pdf->SetKeyWords(
					$outputlangs->convToOutputCharset($object->ref)." ".$outputlangs->transnoentities("InterventionCard")
				);
				if ($conf->global->MAIN_DISABLE_PDF_COMPRESSION) $pdf->SetCompression(false);

				$pdf->SetMargins($this->marge_gauche, $this->marge_haute, $this->marge_droite);   // Left, Top, Right
				$pdf->SetAutoPageBreak(1, 0);

				// New page
				$pdf->AddPage();
				if (! empty($tplidx)) $pdf->useTemplate($tplidx);
				$pagenb++;
				$this->_pagehead($pdf, $object, 1, $outputlangs);
				$pdf->SetTextColor(0, 0, 0);
				$pdf->SetFont('', '', $default_font_size - 1);
				$pdf->MultiCell(0, 3, '');		// Set interline to 3

				$tab_top = 70;
				$tab_top_middlepage = 50;
				$tab_top_newpage = 50;
				$tab_height = 200;
				$tab_height_newpage = 200;
				$tab_height_middlepage = 200;
				$tab_height_endpage = 200;

				// Affiche notes
				if (! empty($object->note_public)) {
					$tab_top = 88;

					$pdf->SetFont('', '', $default_font_size - 1);   // Dans boucle pour gerer multi-page
					$pdf->writeHTMLCell(190, 3, $this->posxdesc-1, $tab_top, dol_htmlentitiesbr($object->note_public), 0, 1);
					$nexY = $pdf->GetY();
					$height_note=$nexY-$tab_top;

					// Rect prend une longueur en 3eme param
					$pdf->SetDrawColor(192, 192, 192);
					$pdf->Rect(
									$this->marge_gauche, $tab_top-1, 
									$this->page_largeur-$this->marge_gauche-$this->marge_droite, $height_note+1
					);

					$tab_height = $tab_height - $height_note;
					$tab_top = $nexY+6;
				} else
					$height_note=0;

				$iniY = $tab_top + 7;
				$curY = $tab_top + 7;
				$nexY = $tab_top + 7;

				//$pdf->line($this->marge_gauche, $nexY, $this->page_largeur-$this->marge_droite, $nexY);
				$pdf->MultiCell(0, 2, '');		// Set interline to 3. Then writeMultiCell must use 3 also.

				//if (GETPOST("displaylist") == "displayfactoryinfo")
				if (true) {	
					$sql.= " SELECT p.ref, p.label, pf.qty, pf.pmp FROM ".MAIN_DB_PREFIX."product as p";
					$sql.= " , ".MAIN_DB_PREFIX."product_factory as pf";
					$sql.= " WHERE pf.fk_product_father= ".$object->id;
					$sql.= " AND pf.fk_product_children = p.rowid";
					$sql.= " ORDER BY ordercomponent";
					
					$tottotal_ht=0;
					
					$resql = $this->db->query($sql);
					if ($resql) {
						$nblines = $this->db->num_rows($resql); 
						if ($nblines > 0) {
							$i=0;
							while ($i < $nblines) {
								$row = $this->db->fetch_object($resql);
								$curY = $nexY;

								// ref product
								$pdf->SetFont('', 'B', $default_font_size - 1);
								$pdf->SetXY($this->marge_gauche, $curY);
								$pdf->writeHTMLCell(30, 3, $this->marge_gauche, $curY, $row->ref, 0, 1, 0, true, "L");

								// label
								$pdf->SetFont('', '', $default_font_size - 1);
								//$pdf->SetXY(30, $curY);
								$pdf->writeHTMLCell(100, 3, 30, $curY, $row->label, 0, 1, 0, true, "L");

								// qty
								$pdf->SetFont('', '', $default_font_size - 1);
								//$pdf->SetXY(130, $curY);
								$pdf->writeHTMLCell(20, 3, 130, $curY, price($row->qty, 0, '', 1, 3), 0, 1, 0, true, "R");

								// price
								$pdf->SetFont('', '', $default_font_size - 1);
								//$pdf->SetXY(150, $curY);
								$pdf->writeHTMLCell(
												25, 3, 150, $curY, 
												price($row->pmp, 0, '', 1, 3), 
												0, 1, 0, true, "R"
								);

								// total price
								$pdf->SetFont('', '', $default_font_size - 1);
								//$pdf->SetXY(170, $curY);
								$pdf->writeHTMLCell(
												30, 3, 169, $curY, 
												price($row->qty * $row->pmp, 0, '', 1, 3), 
												0, 1, 0, true, "R"
								);

								$tottotal_ht+= $row->qty * $row->pmp;

								$nexY+=6;	// Passe espace entre les lignes
		
								// Cherche nombre de lignes a venir pour savoir si place suffisante
								if ($i < ($nblines - 1) && empty($hidedesc)) {
									//on recupere la description du produit suivant
									$follow_descproduitservice = $objectligne->desc;
									//on compte le nombre de ligne afin de verifier la place disponible (largeur de ligne 52 caracteres)
									$nblineFollowDesc = dol_nboflines_bis(
													$follow_descproduitservice, 52, 
													$outputlangs->charset_output
									)*3;
								} else
									$nblineFollowDesc = 0; 	// If it's last line
		
								// Test if a new page is required
								if ($pagenb == 1) {
									$tab_top_in_current_page=$tab_top;
									$tab_height_in_current_page=$tab_height;
								} else {
									$tab_top_in_current_page=$tab_top_newpage;
									$tab_height_in_current_page=$tab_height_middlepage;
								}
								if (($nexY+$nblineFollowDesc) > ($tab_top_in_current_page+$tab_height_in_current_page) 
									&& $i < ($nblines - 1)) {
									if ($pagenb == 1)
										$this->_tableau($pdf, $tab_top, $tab_height + 20, $nexY, $outputlangs);
									else
										$this->_tableau(
														$pdf, $tab_top_newpage, 
														$tab_height_middlepage, $nexY, $outputlangs
										);

									$this->_pagefoot($pdf, $object, $outputlangs);

									// New page
									$pdf->AddPage();
									if (! empty($tplidx)) $pdf->useTemplate($tplidx);
									$pagenb++;
									$this->_pagehead($pdf, $object, 0, $outputlangs);
									$pdf->SetFont('', '', $default_font_size - 1);
									$pdf->MultiCell(0, 3, '');		// Set interline to 3
									$pdf->SetTextColor(0, 0, 0);

									$nexY = $tab_top_newpage + 7;
								}
								$i++;
							}
						}
					}
				}

				// Show square
				if ($pagenb == 1) {
					$this->_tableau($pdf, $tab_top, $tab_height, $nexY, $outputlangs);
					$bottomlasttab=$tab_top + $tab_height + 1;
				} else {
					$this->_tableau($pdf, $tab_top_newpage, $tab_height_newpage, $nexY, $outputlangs);
					$bottomlasttab=$tab_top_newpage + $tab_height_newpage + 1;
				}

				$pdf->line(
								$this->marge_gauche, $tab_top+$tab_height-7, 
								$this->page_largeur-$this->marge_droite, $tab_top+$tab_height-7
				);
				$pdf->writeHTMLCell(
								25, 3, 100, $tab_top+$tab_height-4,
								price($tottotal_ht, 0, '', 1, 3),
								0, 1, 0,true, "R"
				);
				$pdf->writeHTMLCell(
								20, 3, 130, $tab_top+$tab_height-4, 
								$nblines, 0, 1, 0,true, "R"
				);
				$pdf->writeHTMLCell(
								25, 3, 170, $tab_top+$tab_height-4,
								price($tottotal_ht * 1.15, 0, '', 1, 3), 
								0, 1, 0,true, "R"
				);

				$pdf->SetFont('', '', $default_font_size - 1);   // On repositionne la police par defaut

				$this->_pagefoot($pdf, $object, $outputlangs);
				if (method_exists($pdf, 'AliasNbPages')) 
					$pdf->AliasNbPages();

				$pdf->Close();

				$pdf->Output($file,'F');
				if (! empty($conf->global->MAIN_UMASK))
					@chmod($file, octdec($conf->global->MAIN_UMASK));

				return 1;
			} else {
				$this->error=$langs->trans("ErrorCanNotCreateDir", $dir);
				return 0;
			}
		} else {
			$this->error=$langs->trans("ErrorConstantNotDefined", "PRODUCT_OUTPUTDIR");
			return 0;
		}
		$this->error=$langs->trans("ErrorUnknown");
		return 0;   // Erreur par defaut
	}

	/**
	 *   Show table for lines
	 *
	 *   @param		PDF			&$pdf	 		Object PDF
	 *   @param		string		$tab_top		Top position of table
	 *   @param		string		$tab_height		Height of table (rectangle)
	 *   @param		int			$nexY			Y
	 *   @param		Translate	$outputlangs	Langs object
	 *   @return	void
	 */
	function _tableau(&$pdf, $tab_top, $tab_height, $nexY, $outputlangs)
	{
		global $conf;
		$default_font_size = pdf_getPDFFontSize($outputlangs);

		//if (GETPOST("displaylist") == "displayfactoryinfo")
		if (true) {
			$pdf->SetXY($this->marge_gauche, $tab_top+1);
			// ref product
			$pdf->SetFont('', 'B', $default_font_size - 1);
			$pdf->writeHTMLCell(0, 3, $this->marge_gauche, $tab_top+1, $outputlangs->transnoentities("Ref"), 0, 1, 0, true, "L");

			// label
			$pdf->line(30-1, $tab_top, 30-1, $tab_top + $tab_height-7);
			$pdf->SetXY(30, $tab_top+1);
			$pdf->writeHTMLCell(0, 3, 30, $tab_top+1, $outputlangs->transnoentities("Label"), 0, 1, 0, true, "L");

			// qty
			$pdf->line(130-1, $tab_top, 130-1, $tab_top + $tab_height+3);
			$pdf->SetXY(130, $tab_top+1);
			$pdf->writeHTMLCell(20, 3, 130, $tab_top+1, $outputlangs->transnoentities("Qty"), 0, 1, 0, true, "C");

			// price
			$pdf->line(150+1, $tab_top, 150+1, $tab_top + $tab_height+3);
			$pdf->SetXY(150, $tab_top+1);
			$pdf->writeHTMLCell(25, 3, 150, $tab_top+1, $outputlangs->transnoentities("Price"), 0, 1, 0, true, "R");

			// total price
			$pdf->line(175-1, $tab_top, 175-1, $tab_top + $tab_height-7);
			//$pdf->SetXY(165, $tab_top+1);
			$pdf->writeHTMLCell(30, 3, 169, $tab_top+1, $outputlangs->transnoentities("Total"), 0, 1, 0, true, "R");

			$pdf->line($this->marge_gauche, $tab_top + 5, $this->page_largeur-$this->marge_droite, $tab_top + 5);
			$pdf->Rect(
							$this->marge_gauche, $tab_top, 
							($this->page_largeur-$this->marge_gauche-$this->marge_droite), $tab_height+3
			);
		}
	}

	/**
	 *  Show top header of page.
	 *
	 *  @param	PDF			&$pdf	 		Object PDF
	 *  @param  Object		$object	 	Object to show
	 *  @param  int			$showaddress	0=no, 1=yes
	 *  @param  Translate	$outputlangs	Object lang for output
	 *  @return	void
	 */
	function _pagehead(&$pdf, $object, $showaddress, $outputlangs)
	{
		global $conf, $langs;
		$default_font_size = pdf_getPDFFontSize($outputlangs);

		$outputlangs->load("main");
		$outputlangs->load("dict");
		$outputlangs->load("companies");

		pdf_pagehead($pdf, $outputlangs, $this->page_hauteur);

		//Affiche le filigrane brouillon - Print Draft Watermark
		if ($object->statut==0 && (! empty($conf->global->FICHINTER_DRAFT_WATERMARK)) )
			pdf_watermark(
							$pdf, $outputlangs, 
							$this->page_hauteur, $this->page_largeur, 
							'mm', $conf->global->FICHINTER_DRAFT_WATERMARK
			);

		//Prepare la suite
		$pdf->SetTextColor(0, 0, 60);
		$pdf->SetFont('', 'B', $default_font_size + 3);

		$posx=$this->page_largeur-$this->marge_droite-100;
		$posy=$this->marge_haute;

		$pdf->SetXY($this->marge_gauche, $posy);

		// Logo
		$logo=$conf->mycompany->dir_output.'/logos/'.$this->emetteur->logo;
		
		if ($this->emetteur->logo) {
			if (is_readable($logo)) {
				$height=pdf_getHeightForLogo($logo);
				$pdf->Image($logo, $this->marge_gauche, $posy, 0, $height);	// width=0 (auto)
			} else {
				$pdf->SetTextColor(200, 0, 0);
				$pdf->SetFont('', 'B', $default_font_size - 2);
				$pdf->MultiCell(100, 3, $outputlangs->transnoentities("ErrorLogoFileNotFound", $logo), 0, 'L');
				$pdf->MultiCell(100, 3, $outputlangs->transnoentities("ErrorGoToGlobalSetup"), 0, 'L');
			}
		} else {
			$text=$this->emetteur->name;
			$pdf->MultiCell(100, 4, $outputlangs->convToOutputCharset($text), 0, 'L');
		}

		$width=80; 
		$cssclass='photoref';
		$showimage=$object->is_photo_available($conf->product->multidir_output[$object->entity]);
		$maxvisiblephotos=(isset($conf->global->PRODUCT_MAX_VISIBLE_PHOTO)?$conf->global->PRODUCT_MAX_VISIBLE_PHOTO:5);
		if ($showimage)
			$imgphoto= $object->liste_photos($conf->product->multidir_output[$object->entity], 0);

		$pdf->SetFont('', 'B', $default_font_size + 3);
		$pdf->SetXY($posx, $posy);
		$pdf->SetTextColor(0, 0, 60);
		$title=$outputlangs->transnoentities("ProductCard");
		$pdf->MultiCell(100, 4, $title, '', 'R');

		$pdf->SetFont('', 'B', $default_font_size + 2);

		// ref du produit + num immocompta si saisie
		$posy+=10;
		$posx=100;
		$pdf->SetXY($posx, $posy);
		$pdf->SetTextColor(0, 0, 60);
		$refproduct=$object->ref;
		$pdf->MultiCell(100, 4, $outputlangs->transnoentities("Ref")." : " . $refproduct, '', 'L');

		$pdf->SetFont('', '', $default_font_size);
		$posx=120;
		$posy+=6;
		$pdf->SetXY($posx, $posy);
		$pdf->MultiCell(100, 1, $object->label , '', 'L');

		$pdf->SetFont('', '', $default_font_size);
		$posx=10;
		$posy+=6;
		$pdf->SetXY($posx, $posy);
		$pdf->MultiCell(100, 1, $object->description, '', 'L');

		// dates de l'équipement si renseignée	$posy+=10;
		$posx=100;
		$pdf->SetXY($posx, $posy);
		$pdf->MultiCell(
						100, 3, $outputlangs->transnoentities("DateCreate")." : " . dol_print_date(
										$object->date_creation, "day", false, $outputlangs, true
						), '', 'L'
		);

		$posy=50;
		$posx=100;
		$pdf->SetXY($posx, $posy);
		$pdf->MultiCell(100, 3, $outputlangs->transnoentities("ToSell")." : ".yn($object->status), '', 'L');
		$posy+=4;
		$posx=100;
		$pdf->SetXY($posx, $posy);
		$pdf->MultiCell(100, 3, $outputlangs->transnoentities("ToBuy")." : ".yn($object->status_buy), '', 'L');
		$posy+=4;
		$posx=100;
		$pdf->SetXY($posx, $posy);
		$pdf->MultiCell(100, 3, $outputlangs->transnoentities("ToBatch")." : ".yn($object->status_batch), '', 'L');
	}

	/**
	 *		Show footer of page. Need this->emetteur object
	*
	 *		@param	PDF			&$pdf	 			PDF
	 *		@param	Object		$object				Object to show
	 *		@param	Translate	$outputlangs		Object lang for output
	 *		@return	void
	 */
	function _pagefoot(&$pdf, $object, $outputlangs)
	{
		return pdf_pagefoot(
						$pdf, $outputlangs, 'PRODUCT_FREE_TEXT', 
						$this->emetteur, $this->marge_basse, $this->marge_gauche, $this->page_hauteur, 
						$object
		);
	}
}