<?php
/* Copyright (C) 2003      Rodolphe Quiedeville <rodolphe@quiedeville.org>
 * Copyright (C) 2004-2009 Laurent Destailleur  <eldy@users.sourceforge.net>
 * Copyright (C) 2012-2019 Jean-François Ferry    <hello@librethic.io>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/**
     * \defgroup   domain     Module Domain
 *      \brief      Module to manage domain
 */

/**
 *      \file       includes/modules/modDomain.class.php
 *      \ingroup    domain
 *      \brief      Description and activation file for module Domain
 *        \version    $Id: modMyModule.class.php,v 1.32 2009/04/20 21:10:00 eldy Exp $
 */
require_once DOL_DOCUMENT_ROOT . "/core/modules/DolibarrModules.class.php";

/**
     * \class      modDomain
 *      \brief      Description and activation class for module Domain
 */
class modDomain extends DolibarrModules
{
    /**
     *   \brief      Constructor. Define names, constants, directories, boxes, permissions
     *   \param      DB      Database handler
     */
    public function __construct($db)
    {
        global $langs;
        $this->db = $db;

        // Id for module (must be unique).
        // Use here a free id (See in Home -> System information -> Dolibarr for list of used modules id).
        $this->numero = 1664;
        // Key text used to identify module (for permissions, menus, etc...)
        $this->rights_class = 'domain';

        $this->editor_name = "Librethic";
        $this->editor_url = "https://librethic.io";

        // Family can be 'crm','financial','hr','projects','products','ecm','technic','other'
        // It is used to group modules in module setup page
        $this->family = "hosting";
        $this->module_position = '02';
        
        // Module label (no space allowed), used if translation string 'ModuleXXXName' not found (where XXX is value of numeric property 'numero' of module)
        $this->name = preg_replace('@^mod@i', '', get_class($this));
        // Module description, used if translation string 'ModuleXXXDesc' not found (where XXX is value of numeric property 'numero' of module)
        $this->description = "Manage internet domains with hosting module.";
        // Possible values for version are: 'development', 'experimental', 'dolibarr' or version
        $this->version = '2.0.10+9';
        // Key used in llx_const table to save module status enabled/disabled (where MYMODULE is value of property name of module in uppercase)
        $this->const_name = 'MAIN_MODULE_' . strtoupper($this->name);
        
        // Name of png file (without png) used for this module.
        // Png file must be in theme/yourtheme/img directory under name object_pictovalue.png.
        $this->picto = 'domain@domain';

        // Data directories to create when module is enabled.
        $this->dirs = array();
        //$this->dirs[0] = DOL_DATA_ROOT.'/mymodule;
        //$this->dirs[1] = DOL_DATA_ROOT.'/mymodule/temp;

        // Config pages. Put here list of php page names stored in admmin directory used to setup module.
        $this->config_page_url = array("admin_domain.php@domain");

        // Dependencies
        $this->depends = array('modHosting'); // List of modules id that must be enabled if this module is enabled
        $this->requiredby = array(); // List of modules id to disable if this one is disabled
        $this->phpmin = array(5, 6); // Minimum version of PHP required by module
        $this->need_dolibarr_version = array(6, 0); // Minimum version of Dolibarr required by module
        $this->langfiles = array("hosting@hosting", "domain@domain");

        // Constants
        $this->const = array(); // List of particular constants to add when module is enabled
        //Example: $this->const=array(0=>array('MODULE_MY_NEW_CONST1','chaine','myvalue','This is a constant to add',0),
        //                            1=>array('MODULE_MY_NEW_CONST2','chaine','myvalue','This is another constant to add',0) );
        $this->triggers = 1;
        // Onglets
        $this->tabs = array(
            'hosting:tabDomainManagement:Domains:domain@domain:$user->rights->hosting->read:/domain/domain_list.php?hostid=__ID__',
        );
        // where entity can be
        // 'thirdparty'       to add a tab in third party view
        // 'intervention'     to add a tab in intervention view
        // 'supplier_order'   to add a tab in supplier order view
        // 'supplier_invoice' to add a tab in supplier invoice view
        // 'invoice'          to add a tab in customer invoice view
        // 'order'            to add a tab in customer order view
        // 'product'          to add a tab in product view
        // 'propal'           to add a tab in propal view
        // 'member'           to add a tab in fundation member view

        // Defined all module parts (triggers, login, substitutions, menus, css, etc...)
        // for default path (eg: /mymodule/core/xxxxx) (0=disable, 1=enable)
        // for specific path of parts (eg: /mymodule/core/modules/barcode)
        // for specific css file (eg: /mymodule/css/mymodule.css.php)
        //$this->module_parts = array(
        //                            'triggers' => 0,                                     // Set this to 1 if module has its own trigger directory (core/triggers)
        //                            'login' => 0,                                        // Set this to 1 if module has its own login method directory (core/login)
        //                            'substitutions' => 0,                                // Set this to 1 if module has its own substitution function file (core/substitutions)
        //                            'menus' => 0,                                        // Set this to 1 if module has its own menus handler directory (core/menus)
        //                            'theme' => 0,                                        // Set this to 1 if module has its own theme directory (theme)
        //                            'tpl' => 0,                                          // Set this to 1 if module overwrite template dir (core/tpl)
        //                            'barcode' => 0,                                      // Set this to 1 if module has its own barcode directory (core/modules/barcode)
        //                            'models' => 0,                                       // Set this to 1 if module has its own models directory (core/modules/xxx)
        //                            'css' => array('/mymodule/css/mymodule.css.php'),    // Set this to relative path of css file if module has its own css file
        //                            'js' => array('/mymodule/js/mymodule.js'),          // Set this to relative path of js file if module must load a js on all pages
        //                            'hooks' => array('hookcontext1','hookcontext2')      // Set here all hooks context managed by module
        //                            'dir' => array('output' => 'othermodulename'),      // To force the default directories names
        //                            'workflow' => array('WORKFLOW_MODULE1_YOURACTIONTYPE_MODULE2'=>array('enabled'=>'! empty($conf->module1->enabled) && ! empty($conf->module2->enabled)', 'picto'=>'yourpicto@mymodule')) // Set here all workflow context managed by module
        //                        );
        $this->module_parts = array(
            'hooks' => array('searchform', 'commonobject'), // Set here all hooks context managed by module
            'models' => 1,
            'triggers' => 1,
            'tpl' => 0
        );

        $this->cronjobs = array(
            0=>array('label'=>'Update domains database from WHOIS', 'jobtype'=>'method', 'class'=>'/domain/class/DomainCron.class.php', 'objectname'=>'DomainCron', 'method'=>'updateDomainsFromWhois', 'parameters'=>'', 'comment'=>'Fetch datas from WHOIS and update domains in database', 'frequency'=> 1, 'unitfrequency'=>86400, 'status'=>1, 'test'=>true)
        );
        // Boxes
        $this->boxes = array(); // List of boxes
        $r = 0;

        // Add here list of php file(s) stored in includes/boxes that contains class to show a box.
        // Example:
        //$this->boxes[$r][1] = "myboxa.php";
        //$r++;
        //$this->boxes[$r][1] = "myboxb.php";
        //$r++;

        // Permissions
        $this->rights = array(); // Permission array used by this module
        $r = 0;

        $r++;
        $this->rights[$r][0] = 16641;
        $this->rights[$r][1] = 'Voir les domaines';
        $this->rights[$r][3] = 1;
        $this->rights[$r][4] = 'read';

        $r++;
        $this->rights[$r][0] = 16642;
        $this->rights[$r][1] = 'Ajouter/Editer les domains';
        $this->rights[$r][3] = 0;
        $this->rights[$r][4] = 'add';

        $r++;
        $this->rights[$r][0] = 16643;
        $this->rights[$r][1] = 'Supprimer les domaines';
        $this->rights[$r][3] = 0;
        $this->rights[$r][4] = 'delete';

        $r++;
        $this->rights[$r][0] = 16644;
        $this->rights[$r][1] = 'Exporter les domaines';
        $this->rights[$r][3] = 0;
        $this->rights[$r][4] = 'export';

        // Main menu entries
        $this->menu = array(); // List of menus to add
        $r = 0;
        $this->menu[$r] = array(
            'fk_menu' => 'fk_mainmenu=hosting',
            'type' => 'left',
            'titre' => 'Domains',
            'mainmenu' => 'hosting',
            'leftmenu' => 'domain',
            'url' => '/domain/domain_list.php',
            'langs' => 'domain@domain',
            'position' => 100,
            'enabled' => '1',
            'perms' => '$user->rights->domain->read',
            'user' => 2,
        );
        $r++;

        $this->menu[$r++] = array(
            'fk_menu' => 'fk_mainmenu=hosting,fk_leftmenu=domain', //On utilise les ancres définis dans le menu parent déclaré au dessus
            'type' => 'left', // Toujours un menu gauche
            'titre' => 'DomainsList',
            'mainmenu' => 'hosting',
            'leftmenu' => '', // On n'indique rien ici car on ne souhaite pas intégrer de sous-menus à ce menu
            'url' => '/domain/domain_list.php',
            'langs' => 'domain@domain',
            'position' => 101,
            'enabled' => '1',
            'perms' => '$user->rights->domain->read',
            'target' => '',
            'user' => 2,
        );
        $this->menu[$r++] = array(
            'fk_menu' => 'fk_mainmenu=hosting,fk_leftmenu=domain', //On utilise les ancres définis dans le menu parent déclaré au dessus
            'type' => 'left', // Toujours un menu gauche
            'titre' => 'DomainAdd',
            'mainmenu' => 'hosting',
            'leftmenu' => '', // On n'indique rien ici car on ne souhaite pas intégrer de sous-menus à ce menu
            'url' => '/domain/domain_card.php?action=create',
            'langs' => 'domain@domain',
            'position' => 102,
            'enabled' => '1',
            'perms' => '$user->rights->domain->add',
            'target' => '',
            'user' => 2,
        );

        $this->menu[$r++] = array(
            'fk_menu' => 'fk_mainmenu=hosting,fk_leftmenu=domain', //On utilise les ancres définis dans le menu parent déclaré au dessus
            'type' => 'left', // Toujours un menu gauche
            'titre' => 'MenuDomainExpiration',
            'mainmenu' => 'hosting',
            'leftmenu' => '', // On n'indique rien ici car on ne souhaite pas intégrer de sous-menus à ce menu
            'url' => '/domain/expirations.php',
            'langs' => 'domain@domain',
            'position' => 103,
            'enabled' => '1',
            'perms' => '$user->rights->domain->read',
            'target' => '',
            'user' => 0,
        );

        // Exports
        $r = 1;

        $this->export_code[$r] = 'domain_' . $r;
        $this->export_label[$r] = 'DomainNames'; // Translation key (used only if key ExportDataset_xxx_z not found)
        $this->export_icon[$r] = 'domain@domain';
        $this->export_permission[$r] = array(array("domain", "export"));
        $this->export_fields_array[$r] = array('d.rowid' => "IdDomain", 'd.ref' => "RefDomain", 'd.label' => "DomainName", 'd.date_creation' => "DateCreation", 'd.date_registrar' => "DateRegistration", 'd.date_expiration' => "DomainExpirationDate", 's.rowid' => "IdCompany", 's.nom' => 'CompanyName', 's2.nom' => "Supplier");
        $this->export_entities_array[$r] = array('d.rowid' => "Domain", 'd.ref' => "Domain",'d.label' => 'Domain', 'd.date_creation' => "Domain", 'd.date_expiration' => "Domain", 'd.date_registrar' => "domain", 's.rowid' => "company", 's.nom' => 'company', 's2.nom' => 'company');
        $this->export_alias_array[$r] = array('d.rowid' => "domainid", 'd.ref' => "domref", 'd.label' => 'dom_name', 'd.date_creation' => "dome_datecreate", 'd.date_expiration' => 'dom_datexp', 'd.date_registrar' => "dom_datreg", 's.rowid' => "socid", 's.nom' => 'soc_name', 's2.nom' => 'sup_name');
        $this->export_sql_start[$r] = 'SELECT DISTINCT ';
        $this->export_sql_end[$r] = ' FROM (' . MAIN_DB_PREFIX . 'domain as d), ' . MAIN_DB_PREFIX . 'societe as s, ' . MAIN_DB_PREFIX . 'societe as s2';
        $this->export_sql_end[$r] .= ' WHERE d.fk_soc = s.rowid AND d.fk_supplier=s2.rowid';
        $r++;
    }

    /**
     *        \brief      Function called when module is enabled.
     *                    The init function add constants, boxes, permissions and menus (defined in constructor) into Dolibarr database.
     *                    It also creates data directories.
     *      \return     int             1 if OK, 0 if KO
     */
    public function init($options = '')
    {
        $sql = array();

        $result = $this->load_tables();

        return $this->_init($sql, $options);
    }

    /**
     *        \brief        Function called when module is disabled.
     *                  Remove from database constants, boxes and permissions from Dolibarr database.
     *                    Data directories are not deleted.
     *      \return     int             1 if OK, 0 if KO
     */
    public function remove($options = '')
    {
        $sql = array();

        return $this->_remove($sql, $options);
    }

    /**
     *        \brief        Create tables and keys required by module
     *                     Files mymodule.sql and mymodule.key.sql with create table and create keys
     *                     commands must be stored in directory /mymodule/sql/
     *                    This function is called by this->init.
     *         \return        int        <=0 if KO, >0 if OK
     */
    public function load_tables()
    {
        return $this->_load_tables('/domain/sql/');
    }
}
